# (c) Copyright 2020-2021. CodeWeavers, Inc.

import cxlog
import cxobjc
import cxproduct
import cxurlget
import cxutils
import distversion
import proxyinfo
import webtoken

import os
import json
import struct

from urllib.parse import urlencode


def get_system_information():
    ''' Put the system information in a dictionary so we can store it in JSON format.'''
    info = {}

    info['Product'] = distversion.PRODUCT_NAME
    info['Version'] = distversion.PUBLIC_VERSION
    info['Install location'] = cxutils.CX_ROOT
    info['Bitness'] = '%s-bit' % (struct.calcsize('P') * 8)
    info['Desktop session'] = os.environ.get('DESKTOP_SESSION', '')

    try:
        from gi.repository import Gdk
        info['Window manager'] = Gdk.Screen.get_default().get_window_manager_name()
    except: # pylint: disable=W0702
        pass

    info['Locale'] = {}
    lang = os.environ.get('LANG', '')
    info['Locale']['Locale'] = lang
    for var in ('LC_ADDRESS', 'LC_COLLATE', 'LC_CTYPE', 'LC_IDENTIFICATION',
                'LC_MONETARY', 'LC_MESSAGES', 'LC_MEASUREMENT', 'LC_NAME',
                'LC_NUMERIC', 'LC_PAPER', 'LC_TELEPHONE', 'LC_TIME', 'LC_ALL'):
        if var in os.environ and os.environ[var] != lang:
            info['Locale'][var] = os.environ[var]

    info['System Information'] = {}

    # System information on Linux
    for fname in ('/etc/os-release', '/etc/lsb-release'):
        if not os.path.exists(fname):
            continue

        with open(fname, 'r', encoding='utf8', errors='surrogateescape') as f:
            for line in f:
                key, _part, value = line.strip().partition('=')
                info['System Information'][key.strip('"')] = value.strip('"')
        break

    # System information on macOS
    cmd = ('system_profiler', '-detailLevel', 'mini', '-timeout', '1', 'SPDisplaysDataType', 'SPHardwareDataType', 'SPSoftwareDataType')
    retcode, out, _err = cxutils.run(cmd, stdout=cxutils.GRAB)
    if retcode == 0:
        for line in out.splitlines():
            key, _part, value = line.strip().partition(':')
            if value:
                info['System Information'][key.strip()] = value.strip()

    return info

def get_platform():
    '''Put the platform in the right format for the REST API. We submit
    ratings from Chrome OS as "Android" for historical reasons'''
    return 'Android' if cxproduct.is_crostini() else distversion.PLATFORM

def get_version():
    '''Put the version number in the right format for the REST API'''
    return '.'.join(distversion.CX_VERSION.split('.')[0:3])

class RatingUtils(cxobjc.Proxy):
    def __init__(self, bottle, appid):
        cxobjc.Proxy.__init__(self)

        self.url = 'https://www.codeweavers.com/bin/rest/rank/' + appid
        self.login_required = False
        self.connection_failed = False
        self.token = None

        # Specific for this application
        self.rating = None
        self.notes = None
        self.name = bottle

    # Special initializer for objc. This must /always/ be called explicitly
    # on the Mac.
    def initWithAppID_(self, appid):
        self = cxobjc.Proxy.nsobject_init(self)
        if self is not None:
            self.__init__('', appid)
        return self

    @cxobjc.namedSelector(b'fetchRating')
    def get_rating(self):
        self.login_required = False
        self.connection_failed = False
        self.token = webtoken.get_token()

        if not self.token:
            self.login_required = True
            return False

        try:
            proxyinfo.install_default_opener()

            data = {'rank_mode': get_platform(),
                    'rank_ver': get_version()}

            cxlog.log('Obtaining a rating with arguments:')
            cxlog.log(data)

            data['token'] = self.token
            url = self.url + '?' + urlencode(data)

            content = cxurlget.fetch_content(url, timeout=5)
            data = json.loads(cxutils.string_to_unicode(content))

            cxlog.log('Obtained rating data:')
            cxlog.log(data)

            if not isinstance(data, dict):
                return False

            self.notes = data.get('rank_notes')
            self.name = data.get('name')
            rating = data.get('medal_id')
            self.rating = int(rating) if rating else None
            return True
        except cxurlget.HttpError as e:
            # An error occurred while looking for a rating (such as no rating
            # being present or the user not having a valid token).
            if e.code == 401:
                self.login_required = True
            cxlog.log(e.reason)
            return False
        except cxurlget.UrlError as e:
            # Errors in communication with the server.
            self.connection_failed = True
            cxlog.warn(e.reason)
            return False

    @cxobjc.namedSelector(b'setRating:Notes:')
    def set_rating(self, rating, notes):
        self.login_required = False
        self.connection_failed = False
        self.token = webtoken.get_token()

        if not self.token:
            self.login_required = True
            return False

        if not rating or rating <= 0 or rating > 5:
            return True

        try:
            proxyinfo.install_default_opener()
            data = {'rank_mode': get_platform(),
                    'rank_ver': get_version(),
                    'medal_id': rating,
                    'rank_notes': notes if notes is not None else "",
                    'rank_sysinfo': json.dumps(get_system_information())}

            cxlog.log('Submitting a rating:')
            cxlog.log(data)

            data['token'] = self.token

            cxurlget.fetch_content(self.url, timeout=5,
                data=data)
            return True
        except cxurlget.HttpError as e:
            # An error occurred while submitting a rating (such as a duplicate rating
            # or the user not having a valid token).
            if e.code == 401:
                self.login_required = True
                cxlog.log(e.reason)
                return None
            cxlog.warn(e.reason)
            return None
        except cxurlget.UrlError as e:
            # Errors in communication with the server.
            self.connection_failed = True
            cxlog.warn(e.reason)
            return False
