﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3CrtRequest.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/BucketLifecycleConfiguration.h>
#include <aws/s3-crt/model/ChecksumAlgorithm.h>
#include <aws/s3-crt/model/TransitionDefaultMinimumObjectSize.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3Crt {
namespace Model {

/**
 */
class PutBucketLifecycleConfigurationRequest : public S3CrtRequest {
 public:
  AWS_S3CRT_API PutBucketLifecycleConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutBucketLifecycleConfiguration"; }

  AWS_S3CRT_API Aws::String SerializePayload() const override;

  AWS_S3CRT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3CRT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  AWS_S3CRT_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;
  AWS_S3CRT_API Aws::String GetChecksumAlgorithmName() const override;
  inline bool RequestChecksumRequired() const override { return true; };

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_S3CRT_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The name of the bucket for which to set the configuration.</p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  PutBucketLifecycleConfigurationRequest& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the algorithm used to create the checksum for the request when you
   * use the SDK. This header will not provide any additional functionality if you
   * don't use the SDK. When you send this header, there must be a corresponding
   * <code>x-amz-checksum</code> or <code>x-amz-trailer</code> header sent.
   * Otherwise, Amazon S3 fails the request with the HTTP status code <code>400 Bad
   * Request</code>. For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/checking-object-integrity.html">Checking
   * object integrity</a> in the <i>Amazon S3 User Guide</i>.</p> <p>If you provide
   * an individual checksum, Amazon S3 ignores any provided
   * <code>ChecksumAlgorithm</code> parameter.</p>
   */
  inline ChecksumAlgorithm GetChecksumAlgorithm() const { return m_checksumAlgorithm; }
  inline bool ChecksumAlgorithmHasBeenSet() const { return m_checksumAlgorithmHasBeenSet; }
  inline void SetChecksumAlgorithm(ChecksumAlgorithm value) {
    m_checksumAlgorithmHasBeenSet = true;
    m_checksumAlgorithm = value;
  }
  inline PutBucketLifecycleConfigurationRequest& WithChecksumAlgorithm(ChecksumAlgorithm value) {
    SetChecksumAlgorithm(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Container for lifecycle rules. You can add as many as 1,000 rules.</p>
   */
  inline const BucketLifecycleConfiguration& GetLifecycleConfiguration() const { return m_lifecycleConfiguration; }
  inline bool LifecycleConfigurationHasBeenSet() const { return m_lifecycleConfigurationHasBeenSet; }
  template <typename LifecycleConfigurationT = BucketLifecycleConfiguration>
  void SetLifecycleConfiguration(LifecycleConfigurationT&& value) {
    m_lifecycleConfigurationHasBeenSet = true;
    m_lifecycleConfiguration = std::forward<LifecycleConfigurationT>(value);
  }
  template <typename LifecycleConfigurationT = BucketLifecycleConfiguration>
  PutBucketLifecycleConfigurationRequest& WithLifecycleConfiguration(LifecycleConfigurationT&& value) {
    SetLifecycleConfiguration(std::forward<LifecycleConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account ID of the expected bucket owner. If the account ID that you
   * provide does not match the actual owner of the bucket, the request fails with
   * the HTTP status code <code>403 Forbidden</code> (access denied).</p>
   * <p>This parameter applies to general purpose buckets only. It is not supported
   * for directory bucket lifecycle configurations.</p>
   */
  inline const Aws::String& GetExpectedBucketOwner() const { return m_expectedBucketOwner; }
  inline bool ExpectedBucketOwnerHasBeenSet() const { return m_expectedBucketOwnerHasBeenSet; }
  template <typename ExpectedBucketOwnerT = Aws::String>
  void SetExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    m_expectedBucketOwnerHasBeenSet = true;
    m_expectedBucketOwner = std::forward<ExpectedBucketOwnerT>(value);
  }
  template <typename ExpectedBucketOwnerT = Aws::String>
  PutBucketLifecycleConfigurationRequest& WithExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    SetExpectedBucketOwner(std::forward<ExpectedBucketOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates which default minimum object size behavior is applied to the
   * lifecycle configuration.</p>  <p>This parameter applies to general purpose
   * buckets only. It is not supported for directory bucket lifecycle
   * configurations.</p>  <ul> <li> <p> <code>all_storage_classes_128K</code>
   * - Objects smaller than 128 KB will not transition to any storage class by
   * default. </p> </li> <li> <p> <code>varies_by_storage_class</code> - Objects
   * smaller than 128 KB will transition to Glacier Flexible Retrieval or Glacier
   * Deep Archive storage classes. By default, all other storage classes will prevent
   * transitions smaller than 128 KB. </p> </li> </ul> <p>To customize the minimum
   * object size for any transition you can add a filter that specifies a custom
   * <code>ObjectSizeGreaterThan</code> or <code>ObjectSizeLessThan</code> in the
   * body of your transition rule. Custom filters always take precedence over the
   * default transition behavior.</p>
   */
  inline TransitionDefaultMinimumObjectSize GetTransitionDefaultMinimumObjectSize() const { return m_transitionDefaultMinimumObjectSize; }
  inline bool TransitionDefaultMinimumObjectSizeHasBeenSet() const { return m_transitionDefaultMinimumObjectSizeHasBeenSet; }
  inline void SetTransitionDefaultMinimumObjectSize(TransitionDefaultMinimumObjectSize value) {
    m_transitionDefaultMinimumObjectSizeHasBeenSet = true;
    m_transitionDefaultMinimumObjectSize = value;
  }
  inline PutBucketLifecycleConfigurationRequest& WithTransitionDefaultMinimumObjectSize(TransitionDefaultMinimumObjectSize value) {
    SetTransitionDefaultMinimumObjectSize(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  PutBucketLifecycleConfigurationRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  PutBucketLifecycleConfigurationRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key,
                                                                    CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;

  ChecksumAlgorithm m_checksumAlgorithm{ChecksumAlgorithm::NOT_SET};

  BucketLifecycleConfiguration m_lifecycleConfiguration;

  Aws::String m_expectedBucketOwner;

  TransitionDefaultMinimumObjectSize m_transitionDefaultMinimumObjectSize{TransitionDefaultMinimumObjectSize::NOT_SET};

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_bucketHasBeenSet = false;
  bool m_checksumAlgorithmHasBeenSet = false;
  bool m_lifecycleConfigurationHasBeenSet = false;
  bool m_expectedBucketOwnerHasBeenSet = false;
  bool m_transitionDefaultMinimumObjectSizeHasBeenSet = false;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
