﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class UpdateManagedInstanceRoleRequest : public SSMRequest {
 public:
  AWS_SSM_API UpdateManagedInstanceRoleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateManagedInstanceRole"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the managed node where you want to update the role.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  UpdateManagedInstanceRoleRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the Identity and Access Management (IAM) role that you want to
   * assign to the managed node. This IAM role must provide AssumeRole permissions
   * for the Amazon Web Services Systems Manager service principal
   * <code>ssm.amazonaws.com</code>. For more information, see <a
   * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/hybrid-multicloud-service-role.html">Create
   * the IAM service role required for Systems Manager in hybrid and multicloud
   * environments</a> in the <i>Amazon Web Services Systems Manager User
   * Guide</i>.</p>  <p>You can't specify an IAM service-linked role for this
   * parameter. You must create a unique role.</p>
   */
  inline const Aws::String& GetIamRole() const { return m_iamRole; }
  inline bool IamRoleHasBeenSet() const { return m_iamRoleHasBeenSet; }
  template <typename IamRoleT = Aws::String>
  void SetIamRole(IamRoleT&& value) {
    m_iamRoleHasBeenSet = true;
    m_iamRole = std::forward<IamRoleT>(value);
  }
  template <typename IamRoleT = Aws::String>
  UpdateManagedInstanceRoleRequest& WithIamRole(IamRoleT&& value) {
    SetIamRole(std::forward<IamRoleT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  Aws::String m_iamRole;
  bool m_instanceIdHasBeenSet = false;
  bool m_iamRoleHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
