/** @file src/team.c %Team routines. */

#include <stdio.h>
#include "types.h"
#include "os/strings.h"

#include "team.h"

#include "opendune.h"
#include "enhancement.h"
#include "house.h"
#include "pool/pool.h"
#include "pool/pool_house.h"
#include "pool/pool_team.h"
#include "timer/timer.h"
#include "tools/random_general.h"

/**
 * Loop over all teams, performing various of tasks.
 */
void GameLoop_Team(void)
{
	PoolFindStruct find;

	if (g_tickTeamGameLoop > g_timerGame) return;
	g_tickTeamGameLoop = g_timerGame + (Tools_Random_256() & 7) + 5;

	g_scriptCurrentObject    = NULL;
	g_scriptCurrentUnit      = NULL;
	g_scriptCurrentStructure = NULL;

	for (Team *t = Team_FindFirst(&find, HOUSE_INVALID);
			t != NULL;
			t = Team_FindNext(&find)) {
		const House *h = House_Get_ByIndex(t->houseID);

		g_scriptCurrentTeam = t;

		if (!h->flags.isAIActive) continue;

		if (t->script.delay != 0) {
			t->script.delay--;
			continue;
		}

		if (!Script_IsLoaded(&t->script)) continue;

		if (!Script_Run(&t->script)) {
			/* ENHANCEMENT -- Dune2 aborts all other teams if one gives a script error. This doesn't seem correct */
			if (g_dune2_enhanced) continue;
			break;
		}
	}
}

/**
 * Create a new Team.
 *
 * @param houseID The House of the new Team.
 * @param teamActionType The teamActionType of the new Team.
 * @param movementType The movementType of the new Team.
 * @param minMembers The minimum amount of members in the new Team.
 * @param maxMembers The maximum amount of members in the new Team.
 * @return The new created Team, or NULL if something failed.
 */
Team *Team_Create(uint8 houseID, uint8 teamActionType, uint8 movementType, uint16 minMembers, uint16 maxMembers)
{
	Team *t;

	t = Team_Allocate(0xFFFF);

	if (t == NULL) return NULL;
	t->flags.used  = true;
	t->houseID     = houseID;
	t->action      = teamActionType;
	t->actionStart = teamActionType;
	t->movementType = movementType;
	t->minMembers  = minMembers;
	t->maxMembers  = maxMembers;

	Script_Reset(&t->script, g_scriptTeam);
	Script_Load(&t->script, teamActionType);

	t->script.delay = 0;

	return t;
}

/**
 * Convert the name of a team action to the type value of that team action, or
 *  TEAM_ACTION_INVALID if not found.
 */
uint8 Team_ActionStringToType(const char *name)
{
	uint8 type;
	if (name == NULL) return TEAM_ACTION_INVALID;

	for (type = 0; type < TEAM_ACTION_MAX; type++) {
		if (strcasecmp(g_table_teamActionName[type], name) == 0) return type;
	}

	return TEAM_ACTION_INVALID;
}
