/*
 * Copyright (C) 2004 John Ellis
 * Copyright (C) 2008 - 2016 The Geeqie Team
 *
 * Author: John Ellis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "ui-bookmark.h"

#include <array>
#include <cstdlib>
#include <cstring>

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk/gdk.h>
#include <glib-object.h>
#include <pango/pango.h>

#include "compat.h"
#include "history-list.h"
#include "intl.h"
#include "layout.h"
#include "main-defines.h"
#include "misc.h"
#include "pixbuf-util.h"
#include "ui-file-chooser.h"
#include "ui-fileops.h"
#include "ui-menu.h"
#include "ui-misc.h"
#include "ui-tabcomp.h"
#include "ui-utildlg.h"
#include "uri-utils.h"

/*
 *-----------------------------------------------------------------------------
 * bookmarks
 *-----------------------------------------------------------------------------
 */

#define BOOKMARK_DATA_KEY "bookmarkdata"
#define MARKER_PATH "[path]"
#define MARKER_ICON "[icon]"

namespace
{

struct BookButtonData
{
	GtkWidget *button;

	gchar *key;
	gchar *name;
	gchar *path;
	gchar *icon;
	gchar *parent;
};

struct BookMarkData
{
	GtkWidget *widget;
	GtkWidget *box;
	gchar *key;

	BookmarkSelectFunc select_func;

	gboolean no_defaults;
	gboolean editable;
	gboolean only_directories;

	BookButtonData *active_button;
};

struct BookPropData
{
	GtkWidget *name_entry;
	GtkWidget *path_entry;
	GtkWidget *icon_entry;

	BookButtonData *bb;
};

enum {
	TARGET_URI_LIST,
	TARGET_X_URL,
	TARGET_TEXT_PLAIN
};

constexpr std::array<GtkTargetEntry, 3> bookmark_drop_types{{
	{ const_cast<gchar *>("text/uri-list"), 0, TARGET_URI_LIST },
	{ const_cast<gchar *>("x-url/http"),    0, TARGET_X_URL },
	{ const_cast<gchar *>("_NETSCAPE_URL"), 0, TARGET_X_URL }
}};

constexpr std::array<GtkTargetEntry, 2> bookmark_drag_types{{
	{ const_cast<gchar *>("text/uri-list"), 0, TARGET_URI_LIST },
	{ const_cast<gchar *>("text/plain"),    0, TARGET_TEXT_PLAIN }
}};

GList *bookmark_widget_list = nullptr;
GList *bookmark_default_list = nullptr;

const gchar *bookmark_icon(const gchar *path)
{
	if (!isfile(path)) return nullptr;

	g_autofree gchar *real_path = realpath(path, nullptr);

	return strstr(real_path, get_collections_dir()) ? PIXBUF_INLINE_COLLECTION : GQ_ICON_FILE;
}

} // namespace

static void bookmark_populate_all(const gchar *key);


static BookButtonData *bookmark_from_string(const gchar *text)
{
	BookButtonData *b;
	const gchar *path_ptr;
	const gchar *icon_ptr;

	b = g_new0(BookButtonData, 1);

	if (!text)
		{
		b->name = g_strdup(_("New Bookmark"));
		b->path = g_strdup(homedir());
		b->key = nullptr;
		return b;
		}

	b->key = g_strdup(text);

	path_ptr = strstr(text, MARKER_PATH);
	icon_ptr = strstr(text, MARKER_ICON);

	if (path_ptr && icon_ptr && icon_ptr < path_ptr)
		{
		log_printf("warning, bookmark icon must be after path\n");
		return nullptr;
		}

	if (path_ptr)
		{
		static const size_t marker_path_len = strlen(MARKER_PATH);
		gint l;

		l = path_ptr - text;
		b->name = g_strndup(text, l);
		path_ptr += marker_path_len;
		if (icon_ptr)
			{
			l = icon_ptr - path_ptr;
			b->path = g_strndup(path_ptr, l);
			}
		else
			{
			b->path = g_strdup(path_ptr);
			}
		}
	else
		{
		b->name = g_strdup(text);
		b->path = g_strdup("");
		}

	if (icon_ptr)
		{
		static const size_t marker_icon_len = strlen(MARKER_ICON);
		b->icon = g_strdup(icon_ptr + marker_icon_len);
		}

	return b;
}

static void bookmark_free(BookButtonData *b)
{
	if (!b) return;

	g_free(b->name);
	g_free(b->path);
	g_free(b->icon);
	g_free(b->key);
	g_free(b->parent);
	g_free(b);
}

static gchar *bookmark_string(const gchar *name, const gchar *path, const gchar *icon)
{
	if (!name) name = _("New Bookmark");

	if (icon)
		{
		return g_strdup_printf("%s" MARKER_PATH "%s" MARKER_ICON "%s", name, path, icon);
		}

	return g_strdup_printf("%s" MARKER_PATH "%s", name, path);
}

static void bookmark_select_cb(GtkWidget *button, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);
	BookButtonData *b;

	b = static_cast<BookButtonData *>(g_object_get_data(G_OBJECT(button), "bookbuttondata"));
	if (!b) return;

	if (bm->select_func) bm->select_func(b->path);
}

static void bookmark_edit_destroy_cb(GtkWidget *, gpointer data)
{
	auto p = static_cast<BookPropData *>(data);

	bookmark_free(p->bb);
	g_free(p);
}

static void bookmark_edit_ok_cb(GenericDialog *, gpointer data)
{
	auto p = static_cast<BookPropData *>(data);
	const gchar *name;
	const gchar *icon;

	name = gq_gtk_entry_get_text(GTK_ENTRY(p->name_entry));
	g_autofree gchar *path = remove_trailing_slash(gq_gtk_entry_get_text(GTK_ENTRY(p->path_entry)));
	icon = gq_gtk_entry_get_text(GTK_ENTRY(p->icon_entry));

	g_autofree gchar *new_string = bookmark_string(name, path, icon);

	if (p->bb->key)
		{
		history_list_item_change(p->bb->parent, p->bb->key, new_string);
		}
	else
		{
		history_list_add_to_key(p->bb->parent, new_string, 0);
		}

	if (path && path[0] != '\0') tab_completion_append_to_history(p->path_entry, path);
	if (icon && icon[0] != '\0') tab_completion_append_to_history(p->icon_entry, icon);

	bookmark_populate_all(p->bb->parent);
}

/* simply pass NULL for text to turn this into a 'new bookmark' dialog */

static void bookmark_edit(const gchar *key, const gchar *text, GtkWidget *parent)
{
	BookPropData *p;
	GtkWidget *table;
	const gchar *icon;

	if (!key) key = "bookmarks";

	p = g_new0(BookPropData, 1);

	p->bb = bookmark_from_string(text);
	p->bb->parent = g_strdup(key);

	GenericDialog *gd = generic_dialog_new(_("Edit Bookmark"), "bookmark_edit",
	                                       parent, TRUE,
	                                       generic_dialog_dummy_cb, p);
	g_signal_connect(G_OBJECT(gd->dialog), "destroy",
			 G_CALLBACK(bookmark_edit_destroy_cb), p);

	generic_dialog_add_message(gd, nullptr, _("Edit Bookmark"), nullptr, FALSE);

	generic_dialog_add_button(gd, GQ_ICON_OK, "OK",
				  bookmark_edit_ok_cb, TRUE);

	table = pref_table_new(gd->vbox, 3, 2, FALSE, TRUE);
	pref_table_label(table, 0, 0, _("Name:"), GTK_ALIGN_END);

	p->name_entry = gtk_entry_new();
	gtk_widget_set_size_request(p->name_entry, 300, -1);
	if (p->bb->name) gq_gtk_entry_set_text(GTK_ENTRY(p->name_entry), p->bb->name);
	gq_gtk_grid_attach_default(GTK_GRID(table), p->name_entry, 1, 2, 0, 1);
	generic_dialog_attach_default(gd, p->name_entry);
	gtk_widget_show(p->name_entry);

	pref_table_label(table, 0, 1, _("Path:"), GTK_ALIGN_END);

	p->path_entry = tab_completion_new_with_history(nullptr, p->bb->path, "bookmark_path", -1);
	tab_completion_add_select_button(p->path_entry, nullptr, TRUE, nullptr, nullptr, nullptr);
	gq_gtk_grid_attach_default(GTK_GRID(table), tab_completion_get_box(p->path_entry), 1, 2, 1, 2);
	generic_dialog_attach_default(gd, p->path_entry);

	pref_table_label(table, 0, 2, _("Icon:"), GTK_ALIGN_END);

	icon = p->bb->icon;
	if (!icon) icon = "";
	p->icon_entry = tab_completion_new_with_history(nullptr, icon, "bookmark_icons", -1);
	tab_completion_add_select_button(p->icon_entry, _("Select icon"), FALSE, nullptr, nullptr, nullptr);
	gq_gtk_grid_attach_default(GTK_GRID(table), tab_completion_get_box(p->icon_entry), 1, 2, 2, 3);
	generic_dialog_attach_default(gd, p->icon_entry);

	gtk_widget_show(gd->dialog);
}

static void bookmark_move(BookMarkData *bm, GtkWidget *button, gint direction)
{
	if (!bm->editable) return;

	auto *b = static_cast<BookButtonData *>(g_object_get_data(G_OBJECT(button), "bookbuttondata"));
	if (!b) return;

	g_autoptr(GList) list = gq_gtk_widget_get_children(GTK_WIDGET(bm->box));

	gint p = g_list_index(list, button);
	if (p < 0 || p + direction < 0) return;

	gchar *key_holder = bm->key;
	bm->key = const_cast<gchar *>("_TEMPHOLDER");
	history_list_item_move(key_holder, b->key, -direction);
	bookmark_populate_all(key_holder);
	bm->key = key_holder;

	gtk_box_reorder_child(GTK_BOX(bm->box), button, p + direction);
}

static void bookmark_menu_prop_cb(GtkWidget *widget, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);

	if (!bm->active_button) return;

	bookmark_edit(bm->key, bm->active_button->key, widget);
}

template<gint direction>
static void bookmark_menu_move_cb(GtkWidget *, gpointer data)
{
	auto *bm = static_cast<BookMarkData *>(data);

	if (!bm->active_button) return;

	bookmark_move(bm, bm->active_button->button, direction);
}

static void bookmark_menu_remove_cb(GtkWidget *, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);

	if (!bm->active_button) return;

	history_list_item_remove(bm->key, bm->active_button->key);
	bookmark_populate_all(bm->key);
}

static void bookmark_menu_popup(BookMarkData *bm, GtkWidget *button, gint, guint32, gboolean local)
{
	GtkWidget *menu;
	BookButtonData *b;

	b = static_cast<BookButtonData *>(g_object_get_data(G_OBJECT(button), "bookbuttondata"));
	if (!b) return;

	bm->active_button = b;

	menu = popup_menu_short_lived();
	menu_item_add_icon_sensitive(menu, _("_Properties..."), PIXBUF_INLINE_ICON_PROPERTIES, bm->editable,
		      G_CALLBACK(bookmark_menu_prop_cb), bm);
	menu_item_add_icon_sensitive(menu, _("Move _up"), GQ_ICON_GO_UP, bm->editable,
	                             G_CALLBACK(bookmark_menu_move_cb<-1>), bm);
	menu_item_add_icon_sensitive(menu, _("Move _down"), GQ_ICON_GO_DOWN, bm->editable,
	                             G_CALLBACK(bookmark_menu_move_cb<1>), bm);
	menu_item_add_icon_sensitive(menu, _("_Remove"), GQ_ICON_REMOVE, bm->editable,
		      G_CALLBACK(bookmark_menu_remove_cb), bm);

	if (local)
		{
		gtk_menu_popup_at_widget(GTK_MENU(menu), button, GDK_GRAVITY_NORTH_EAST, GDK_GRAVITY_CENTER, nullptr);
		}
	else
		{
		gtk_menu_popup_at_pointer(GTK_MENU(menu), nullptr);
		}
}

static gboolean bookmark_press_cb(GtkWidget *button, GdkEventButton *event, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);

	if (event->button != GDK_BUTTON_SECONDARY) return FALSE;

	bookmark_menu_popup(bm, button, event->button, event->time, FALSE);

	return TRUE;
}

static gboolean bookmark_keypress_cb(GtkWidget *button, GdkEventKey *event, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);

	switch (event->keyval)
		{
		case GDK_KEY_F10:
			if (!(event->state & GDK_CONTROL_MASK)) return FALSE;
			/* fall through */
		case GDK_KEY_Menu:
			bookmark_menu_popup(bm, button, 0, event->time, TRUE);
			return TRUE;
			break;
		case GDK_KEY_Up:
			if (event->state & GDK_SHIFT_MASK)
				{
				bookmark_move(bm, button, -1);
				return TRUE;
				}
			break;
		case GDK_KEY_Down:
			if (event->state & GDK_SHIFT_MASK)
				{
				bookmark_move(bm, button, 1);
				return TRUE;
				}
			break;
		default:
			break;
		}

	return FALSE;
}

#if !HAVE_GTK4
static void bookmark_drag_set_data(GtkWidget *button,
				   GdkDragContext *context, GtkSelectionData *selection_data,
				   guint, guint, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);
	BookButtonData *b;
	GList *list = nullptr;

	return;
	if (gdk_drag_context_get_dest_window(context) == gtk_widget_get_window(bm->widget)) return;

	b = static_cast<BookButtonData *>(g_object_get_data(G_OBJECT(button), "bookbuttondata"));
	if (!b) return;

	list = g_list_append(list, b->path);
	uri_selection_data_set_uris_from_pathlist(selection_data, list);
	g_list_free(list);
}

static void bookmark_drag_begin(GtkWidget *button, GdkDragContext *context, gpointer)
{
	GdkPixbuf *pixbuf;
	gint x;
	gint y;
	GtkAllocation allocation;
	GdkSeat *seat;
	GdkDevice *device;

	gtk_widget_get_allocation(button, &allocation);

	pixbuf = gdk_pixbuf_get_from_window(gtk_widget_get_window(button),
					    allocation.x, allocation.y,
					    allocation.width, allocation.height);
	seat = gdk_display_get_default_seat(gdk_window_get_display(gtk_widget_get_window(button)));
	device = gdk_seat_get_pointer(seat);
	get_pointer_position(button, device, &x, &y, nullptr);

	gtk_drag_set_icon_pixbuf(context, pixbuf,
				 x - allocation.x, y - allocation.y);
	g_object_unref(pixbuf);
}
#endif

static gboolean bookmark_path_tooltip_cb(GtkWidget *button, gpointer)
{
	BookButtonData *b;

	b = static_cast<BookButtonData *>(g_object_get_data(G_OBJECT(button), "bookbuttondata"));
	gtk_widget_set_tooltip_text(button, b->path);

	return FALSE;
}

static void bookmark_populate(BookMarkData *bm)
{
	static const auto destroy_widget = [](GtkWidget *widget, gpointer)
	{
		gq_gtk_widget_destroy(widget);
	};
	gtk_container_foreach(GTK_CONTAINER(bm->box), destroy_widget, nullptr);

	if (!bm->no_defaults && !history_list_get_by_key(bm->key))
		{
		if (!bookmark_default_list)
			{
			g_autofree gchar *home_buf = bookmark_string(_("Home"), homedir(), nullptr);
			history_list_add_to_key(bm->key, home_buf, 0);

			if (g_strcmp0(bm->key, "shortcuts") != 0)
				{
				g_autofree gchar *buf = bookmark_string(".", history_list_find_last_path_by_key("path_list"), nullptr);
				history_list_add_to_key(bm->key, buf, 0);
				}

			g_autofree gchar *path = g_build_filename(homedir(), "Desktop", NULL);
			if (isname(path))
				{
				g_autofree gchar *buf = bookmark_string(_("Desktop"), path, nullptr);
				history_list_add_to_key(bm->key, buf, 0);
				}
			}

		GList *work = bookmark_default_list;
		while (work && work->next)
			{
			auto *name = static_cast<gchar *>(work->data);
			work = work->next;
			auto *path = static_cast<gchar *>(work->data);
			work = work->next;

			g_autofree gchar *buf = nullptr;
			if (strcmp(name, ".") == 0)
				{
				if (g_strcmp0(bm->key, "shortcuts") == 0) continue;

				buf = bookmark_string(name, history_list_find_last_path_by_key("path_list"), nullptr);
				}
			else
				{
				buf = bookmark_string(name, path, nullptr);
				}
			history_list_add_to_key(bm->key, buf, 0);
			}
		}

	GList *work = history_list_get_by_key(bm->key);
	work = g_list_last(work);
	while (work)
		{
		BookButtonData *b;

		b = bookmark_from_string(static_cast<const gchar *>(work->data));
		if (b)
			{
			if (strcmp(b->name, ".") == 0)
				{
				gchar *buf;

				b->path = g_strdup(history_list_find_last_path_by_key("path_list"));
				buf = bookmark_string(".", b->path, b->icon);
				history_list_item_change("bookmarks", b->key, buf);
				b->key = buf;
				}
			GtkWidget *box;

			b->button = gtk_button_new();
			gtk_button_set_relief(GTK_BUTTON(b->button), GTK_RELIEF_NONE);
			gq_gtk_box_pack_start(GTK_BOX(bm->box), b->button, FALSE, FALSE, 0);
			gtk_widget_show(b->button);

			g_object_set_data_full(G_OBJECT(b->button), "bookbuttondata",
					       b, reinterpret_cast<GDestroyNotify>(bookmark_free));

			box = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, PREF_PAD_BUTTON_GAP);
			gq_gtk_container_add(b->button, box);
			gtk_widget_show(box);

			GtkWidget *image;
			if (b->icon)
				{
				g_autoptr(GdkPixbuf) pixbuf = nullptr;

				if (isfile(b->icon))
					{
					g_autofree gchar *iconl = path_from_utf8(b->icon);
					pixbuf = gdk_pixbuf_new_from_file(iconl, nullptr);
					}
				else
					{
					gint w;
					gint h;

					w = h = 16;
					gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &w, &h);

					pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), b->icon, w, GTK_ICON_LOOKUP_NO_SVG, nullptr);
					}

				if (pixbuf)
					{
					gint w;
					gint h;

					w = h = 16;
					gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &w, &h);

					g_autoptr(GdkPixbuf) scaled = gdk_pixbuf_scale_simple(pixbuf, w, h, GDK_INTERP_BILINEAR);
					image = gtk_image_new_from_pixbuf(scaled);
					}
				else
					{
					image = gtk_image_new_from_icon_name(GQ_ICON_DIRECTORY, GTK_ICON_SIZE_BUTTON);
					}
				}
			else
				{
				image = gtk_image_new_from_icon_name(GQ_ICON_DIRECTORY, GTK_ICON_SIZE_BUTTON);
				}
			gq_gtk_box_pack_start(GTK_BOX(box), image, FALSE, FALSE, 0);
			gtk_widget_show(image);

			pref_label_new(box, b->name);

			g_signal_connect(G_OBJECT(b->button), "clicked",
					 G_CALLBACK(bookmark_select_cb), bm);
			g_signal_connect(G_OBJECT(b->button), "button_press_event",
					 G_CALLBACK(bookmark_press_cb), bm);
			g_signal_connect(G_OBJECT(b->button), "key_press_event",
					 G_CALLBACK(bookmark_keypress_cb), bm);

			gq_gtk_drag_source_set(b->button, GDK_BUTTON1_MASK,
			                    bookmark_drag_types.data(), bookmark_drag_types.size(),
			                    static_cast<GdkDragAction>(GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK));
			gq_drag_g_signal_connect(G_OBJECT(b->button), "drag_data_get",
					 G_CALLBACK(bookmark_drag_set_data), bm);
			gq_drag_g_signal_connect(G_OBJECT(b->button), "drag_begin",
					 G_CALLBACK(bookmark_drag_begin), bm);

			gtk_widget_set_has_tooltip(b->button, TRUE);
			g_signal_connect(G_OBJECT(b->button), "query_tooltip", G_CALLBACK(bookmark_path_tooltip_cb), bm);
			}

		work = work->prev;
		}
}

static void bookmark_populate_all(const gchar *key)
{
	GList *work;

	if (!key) return;

	work = bookmark_widget_list;
	while (work)
		{
		BookMarkData *bm;

		bm = static_cast<BookMarkData *>(work->data);
		work = work->next;

		if (strcmp(bm->key, key) == 0)
			{
			bookmark_populate(bm);
			}
		}
}

#if !HAVE_GTK4
static void bookmark_dnd_get_data(GtkWidget *, GdkDragContext *,
				  gint, gint,
				  GtkSelectionData *selection_data, guint,
				  guint, gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);

	if (!bm->editable) return;

	GList *list = uri_pathlist_from_gtk_selection_data(selection_data);

	for (GList *work = list; work; work = work->next)
		{
		auto path = static_cast<gchar *>(work->data);

		if (bm->only_directories && !isdir(path)) continue;

		g_autofree gchar *buf = bookmark_string(filename_from_path(path), path, bookmark_icon(path));
		history_list_add_to_key(bm->key, buf, 0);
		}

	g_list_free_full(list, g_free);

	bookmark_populate_all(bm->key);
}
#endif

static void bookmark_list_destroy(gpointer data)
{
	auto bm = static_cast<BookMarkData *>(data);

	bookmark_widget_list = g_list_remove(bookmark_widget_list, bm);

	g_free(bm->key);
	g_free(bm);
}

GtkWidget *bookmark_list_new(const gchar *key, const BookmarkSelectFunc &select_func)
{
	BookMarkData *bm;

	if (!key) key = "bookmarks";

	bm = g_new0(BookMarkData, 1);
	bm->key = g_strdup(key);

	bm->select_func = select_func;

	bm->no_defaults = FALSE;
	bm->editable = TRUE;
	bm->only_directories = FALSE;

	GtkWidget *scrolled = gq_gtk_scrolled_window_new(nullptr, nullptr);

	PangoLayout *layout;
	gint width;
	gint height;

	layout = gtk_widget_create_pango_layout(scrolled, "reasonable width");
	pango_layout_get_pixel_size(layout, &width, &height);
	gtk_scrolled_window_set_min_content_width(GTK_SCROLLED_WINDOW(scrolled), width);
	g_object_unref(layout);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	bm->box = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
	gq_gtk_container_add(scrolled, bm->box);
	gtk_widget_show(bm->box);

	bookmark_populate(bm);

	g_object_set_data_full(G_OBJECT(bm->box), BOOKMARK_DATA_KEY, bm, bookmark_list_destroy);
	g_object_set_data(G_OBJECT(scrolled), BOOKMARK_DATA_KEY, bm);
	bm->widget = scrolled;

	gq_gtk_drag_dest_set(scrolled,
	                  static_cast<GtkDestDefaults>(GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP),
	                  bookmark_drop_types.data(), bookmark_drop_types.size(),
	                  static_cast<GdkDragAction>(GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK));
	g_signal_connect(G_OBJECT(scrolled), "drag_data_received",
			 G_CALLBACK(bookmark_dnd_get_data), bm);

	bookmark_widget_list = g_list_append(bookmark_widget_list, bm);

	return scrolled;
}

void bookmark_list_set_key(GtkWidget *list, const gchar *key)
{
	BookMarkData *bm;

	if (!list || !key) return;

	bm = static_cast<BookMarkData *>(g_object_get_data(G_OBJECT(list), BOOKMARK_DATA_KEY));
	if (!bm) return;

	if (bm->key && strcmp(bm->key, key) == 0) return;

	g_free(bm->key);
	bm->key = g_strdup(key);

	bookmark_populate(bm);
}

void bookmark_list_set_no_defaults(GtkWidget *list, gboolean no_defaults)
{
	BookMarkData *bm;

	bm = static_cast<BookMarkData *>(g_object_get_data(G_OBJECT(list), BOOKMARK_DATA_KEY));
	if (!bm) return;

	bm->no_defaults = no_defaults;
}

void bookmark_list_set_editable(GtkWidget *list, gboolean editable)
{
	BookMarkData *bm;

	bm = static_cast<BookMarkData *>(g_object_get_data(G_OBJECT(list), BOOKMARK_DATA_KEY));
	if (!bm) return;

	bm->editable = editable;
}

void bookmark_list_set_only_directories(GtkWidget *list, gboolean only_directories)
{
	BookMarkData *bm;

	bm = static_cast<BookMarkData *>(g_object_get_data(G_OBJECT(list), BOOKMARK_DATA_KEY));
	if (!bm) return;

	bm->only_directories = only_directories;
}

void bookmark_list_add(GtkWidget *list, const gchar *name, const gchar *path)
{
	BookMarkData *bm;

	bm = static_cast<BookMarkData *>(g_object_get_data(G_OBJECT(list), BOOKMARK_DATA_KEY));
	if (!bm) return;

	g_autofree gchar *buf = bookmark_string(name, path, bookmark_icon(path));
	history_list_add_to_key(bm->key, buf, 0);

	bookmark_populate_all(bm->key);
}

/**
 * @brief Allows apps to set up the defaults
 */
void bookmark_add_default(const gchar *name, const gchar *path)
{
	if (!name || !path) return;
	bookmark_default_list = g_list_append(bookmark_default_list, g_strdup(name));
	bookmark_default_list = g_list_append(bookmark_default_list, g_strdup(path));
}

static void bookmark_add_response_cb(GtkFileChooser *chooser, gint response_id, gpointer data)
{
	if (response_id == GTK_RESPONSE_ACCEPT)
		{
		GtkWidget *name_entry = gtk_file_chooser_get_extra_widget(chooser);
		const gchar *name = gtk_entry_get_text(GTK_ENTRY(name_entry));
		gboolean empty_name = (g_strcmp0(name, "") == 0);

		g_autoptr(GFile) file = gtk_file_chooser_get_file(chooser);
		g_autofree gchar *selected_dir = g_file_get_path(file);

		auto *list = static_cast<GtkWidget *>(data);
		bookmark_list_add(list, empty_name ? filename_from_path(selected_dir) : name, selected_dir);
		}

	gq_gtk_widget_destroy(GTK_WIDGET(chooser));
}

void bookmark_add_dialog(const gchar *title, GtkWidget *list)
{
	FileChooserDialogData fcdd{};

	fcdd.action = GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER;
	fcdd.accept_text = _("Open");
	fcdd.data = list;
	fcdd.entry_text = _("Optional name...");
	fcdd.entry_tooltip =  _("Optional alias name for the shortcut.\nThis may be amended or added from the Sort Manager pane.\nIf none given, the basename of the folder is used");
	fcdd.filename = layout_get_path(get_current_layout());
	fcdd.response_callback = G_CALLBACK(bookmark_add_response_cb);
	fcdd.title = title;

	GtkFileChooserDialog *dialog = file_chooser_dialog_new(fcdd);

	gq_gtk_widget_show_all(GTK_WIDGET(dialog));
}

/*
 *-----------------------------------------------------------------------------
 * combo with history key
 *-----------------------------------------------------------------------------
 */

struct HistoryComboData
{
	GtkWidget *combo;
	GtkWidget *entry;
	gchar *history_key;
	gint history_levels;
};

static void history_combo_destroy(gpointer data)
{
	auto hc = static_cast<HistoryComboData *>(data);

	g_free(hc->history_key);
	g_free(data);
}

/* if text is NULL, entry is set to the most recent item */
GtkWidget *history_combo_new(GtkWidget **entry, const gchar *text,
			     const gchar *history_key, gint max_levels)
{
	HistoryComboData *hc;
	GList *work;
	gint n = 0;

	hc = g_new0(HistoryComboData, 1);
	hc->history_key = g_strdup(history_key);
	hc->history_levels = max_levels;

	hc->combo = gtk_combo_box_text_new_with_entry();

	hc->entry = gq_gtk_bin_get_child(GTK_WIDGET(hc->combo));

	g_object_set_data_full(G_OBJECT(hc->combo), "history_combo_data", hc, history_combo_destroy);
	g_object_set_data(G_OBJECT(hc->entry), "history_combo_data", hc);

	work = history_list_get_by_key(hc->history_key);
	while (work)
		{
		gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(hc->combo), static_cast<gchar *>(work->data));
		work = work->next;
		n++;
		}

	if (text)
		{
		gq_gtk_entry_set_text(GTK_ENTRY(hc->entry), text);
		}
	else if (n > 0)
		{
		gtk_combo_box_set_active(GTK_COMBO_BOX(hc->combo), 0);
		}

	if (entry) *entry = hc->entry;
	return hc->combo;
}

/* if text is NULL, current entry text is used
 * widget can be the combo or entry widget
 */
void history_combo_append_history(GtkWidget *widget, const gchar *text)
{
	HistoryComboData *hc;

	hc = static_cast<HistoryComboData *>(g_object_get_data(G_OBJECT(widget), "history_combo_data"));
	if (!hc)
		{
		log_printf("widget is not a history combo\n");
		return;
		}

	g_autofree gchar *new_text = g_strdup(text ? text : gq_gtk_entry_get_text(GTK_ENTRY(hc->entry)));

	if (new_text && new_text[0] != '\0')
		{
		GtkTreeModel *store;
		GList *work;

		history_list_add_to_key(hc->history_key, new_text, hc->history_levels);

		gtk_combo_box_set_active(GTK_COMBO_BOX(hc->combo), -1);

		store = gtk_combo_box_get_model(GTK_COMBO_BOX(hc->combo));
		gtk_list_store_clear(GTK_LIST_STORE(store));

		work = history_list_get_by_key(hc->history_key);
		while (work)
			{
			gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(hc->combo), static_cast<gchar *>(work->data));
			work = work->next;
			}
		}
}
/* vim: set shiftwidth=8 softtabstop=0 cindent cinoptions={1s: */
