// Copyright (c) Meta Platforms, Inc. and affiliates.

#include "openzl/codecs/prefix/decode_prefix_kernel.h"
#include <string.h>

#include "openzl/codecs/common/copy.h"
#include "openzl/common/assertion.h"
#include "openzl/shared/numeric_operations.h"
#include "openzl/shared/overflow.h"
#include "openzl/zl_errors.h"

// Local macro for alignment attributes - specific to this kernel file
#ifdef _MSC_VER
#    define ZL_PREFIX_ALIGNED(n) __declspec(align(n))
#else
#    define ZL_PREFIX_ALIGNED(n) __attribute__((aligned(n)))
#endif

size_t ZS_calcOriginalPrefixSize(
        const uint32_t* const matchSizes,
        size_t const eltWidthsSum,
        size_t const nbElts)
{
    ZL_ASSERT(sizeof(size_t) == 8);
    size_t dstSize = eltWidthsSum;
    for (size_t i = 0; i < nbElts; i++) {
        dstSize += (size_t)matchSizes[i];
    }
    return dstSize;
}

ZL_FORCE_NOINLINE ZL_Report ZS_decodePrefix_fallback(
        uint8_t* prevOutPtr,
        uint8_t* currOutPtr,
        uint32_t* const fieldSizes,
        const uint8_t* suffixes,
        size_t const nbElts,
        const uint32_t* const eltWidths,
        const uint32_t* const matchSizes,
        size_t prevFieldSize)
{
    const uint8_t* currSuffixPtr = suffixes;
    for (size_t i = 0; i < nbElts; ++i) {
        uint32_t const currEltWidth  = eltWidths[i];
        uint32_t const currMatchSize = matchSizes[i];
        ZL_RET_R_IF_GT(corruption, currMatchSize, prevFieldSize);

        // Write the shared prefix
        memcpy(currOutPtr, prevOutPtr, currMatchSize);

        // Write the suffix
        memcpy(currOutPtr + currMatchSize, currSuffixPtr, currEltWidth);

        prevFieldSize = currMatchSize + currEltWidth;
        prevOutPtr    = currOutPtr;
        currOutPtr += prevFieldSize;
        currSuffixPtr += currEltWidth;
        ZL_RET_R_IF(
                corruption,
                ZL_overflowAddU32(currMatchSize, currEltWidth, &fieldSizes[i]));
    }

    return ZL_returnSuccess();
}

#if ZL_HAS_AVX2
#    include <immintrin.h>

// clang-format off
static const uint8_t blendMasks[][sizeof(__m256i)]
        ZL_PREFIX_ALIGNED(32) = {
            { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
        };
// clang-format on

#endif // ZL_HAS_AVX2

ZL_Report ZS_decodePrefix(
        uint8_t* const out,
        uint32_t* const fieldSizes,
        const uint8_t* const suffixes,
        size_t const nbElts,
        const uint32_t* const eltWidths,
        const uint32_t* const matchSizes)
{
    // Calculate the number of safe wildcopies
    size_t nbWildcopies = nbElts;

    for (size_t suffixSum = 0;
         nbWildcopies > 0 && suffixSum < ZS_WILDCOPY_OVERLENGTH;
         --nbWildcopies) {
        suffixSum += eltWidths[nbWildcopies - 1];
    }

    const uint8_t* currSuffixPtr = suffixes;
    uint8_t* prevOutPtr          = out;
    uint8_t* currOutPtr          = out;
    uint32_t prevFieldSize       = 0;

    ZL_STATIC_ASSERT(
            ZS_WILDCOPY_OVERLENGTH >= 32,
            "Ensure that it's safe to load a vector");

#if ZL_HAS_AVX2
    __m256i vec = _mm256_setzero_si256();
#endif // ZL_HAS_AVX2
    for (size_t i = 0; i < nbWildcopies; ++i) {
        uint32_t const currEltWidth  = eltWidths[i];
        uint32_t const currMatchSize = matchSizes[i];
        ZL_RET_R_IF_GT(corruption, currMatchSize, prevFieldSize);

// Write the shared prefix and the suffix
// We branch between two implementations, where one is an optimized version that
// reduces store-forward stalls. This happens while copying the prefix because
// our read may overlap multiple store-forward buffers
#if ZL_HAS_AVX2
        if (currMatchSize <= sizeof(__m256i)) {
            // Blend the previous field with the current suffix offset by
            // `currMatchSize` This won't underflow as any `matchSize` can be
            // represented as a sum of `suffixSize`s
            vec = _mm256_blendv_epi8(
                    vec,
                    _mm256_loadu_si256(
                            (__m256i_u const*)(currSuffixPtr - currMatchSize)),
                    _mm256_load_si256(
                            (__m256i const*)(blendMasks[currMatchSize])));
            _mm256_storeu_si256((__m256i_u*)(currOutPtr), vec);
            if (currMatchSize + currEltWidth > sizeof(__m256i)) {
                uint32_t remSuffix =
                        (currMatchSize + currEltWidth) - sizeof(__m256i);
                ZS_wildcopy(
                        currOutPtr + sizeof(__m256i),
                        currSuffixPtr + (currEltWidth - remSuffix),
                        remSuffix,
                        ZS_wo_no_overlap);
            }
        } else {
            ZS_wildcopy(
                    currOutPtr, prevOutPtr, currMatchSize, ZS_wo_no_overlap);
            ZS_wildcopy(
                    currOutPtr + currMatchSize,
                    currSuffixPtr,
                    currEltWidth,
                    ZS_wo_no_overlap);
        }
#else
        memcpy(currOutPtr, prevOutPtr, currMatchSize);
        ZS_wildcopy(
                currOutPtr + currMatchSize,
                currSuffixPtr,
                currEltWidth,
                ZS_wo_no_overlap);
#endif // ZL_HAS_AVX2

        // Setup for next iteration
        prevFieldSize = currMatchSize + currEltWidth;
        prevOutPtr    = currOutPtr;
        currOutPtr += prevFieldSize;
        currSuffixPtr += currEltWidth;
        ZL_RET_R_IF(
                corruption,
                ZL_overflowAddU32(currMatchSize, currEltWidth, &fieldSizes[i]));
    }

    // Fallback to copy rest of elements
    return ZS_decodePrefix_fallback(
            prevOutPtr,
            currOutPtr,
            fieldSizes + nbWildcopies,
            currSuffixPtr,
            nbElts - nbWildcopies,
            eltWidths + nbWildcopies,
            matchSizes + nbWildcopies,
            prevFieldSize);
}
