/*
 *  $Id: distance.c 28788 2025-11-04 17:13:02Z yeti-dn $
 *  Copyright (C) 2003-2022 Nenad Ocelic, David Necas (Yeti), Petr Klapetek.
 *  E-mail: ocelic@biochem.mpg.de, yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>

enum {
    COLUMN_I, COLUMN_DX, COLUMN_DY, COLUMN_PHI, COLUMN_R, COLUMN_DZ, NCOLUMNS
};

enum {
    PARAM_REPORT_STYLE,
    PARAM_NUMBER_LINES,
    PARAM_HOLD_SELECTION,
};

#define GWY_TYPE_TOOL_DISTANCE            (gwy_tool_distance_get_type())
#define GWY_TOOL_DISTANCE(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), GWY_TYPE_TOOL_DISTANCE, GwyToolDistance))
#define GWY_IS_TOOL_DISTANCE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), GWY_TYPE_TOOL_DISTANCE))
#define GWY_TOOL_DISTANCE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), GWY_TYPE_TOOL_DISTANCE, GwyToolDistanceClass))

typedef struct _GwyToolDistance      GwyToolDistance;
typedef struct _GwyToolDistanceClass GwyToolDistanceClass;

struct _GwyToolDistance {
    GwyPlainTool parent_instance;

    GwyParams *params;

    GtkTreeView *treeview;
    GtkTreeModel *model;
    GwyParamTable *table;

    /* potential class data */
    GwyValueFormat *angle_format;
};

struct _GwyToolDistanceClass {
    GwyPlainToolClass parent_class;
};

static gboolean     module_register                    (void);
static GwyParamDef* define_module_params               (void);
static GType        gwy_tool_distance_get_type         (void)                      G_GNUC_CONST;
static void         gwy_tool_distance_finalize         (GObject *object);
static void         gwy_tool_distance_init_dialog      (GwyToolDistance *tool);
static void         gwy_tool_distance_data_switched    (GwyTool *gwytool,
                                                        GwyDataView *data_view);
static void         gwy_tool_distance_data_changed     (GwyPlainTool *plain_tool);
static void         gwy_tool_distance_selection_changed(GwyPlainTool *plain_tool,
                                                        gint hint);
static void         param_changed                      (GwyToolDistance *tool,
                                                        gint id);
static void         update_headers                     (GwyToolDistance *tool);
static void         render_cell                        (GtkCellLayout *layout,
                                                        GtkCellRenderer *renderer,
                                                        GtkTreeModel *model,
                                                        GtkTreeIter *iter,
                                                        gpointer user_data);
static gchar*       format_report                      (gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Distance measurement tool, measures distances and angles."),
    "Nenad Ocelic <ocelic@biochem.mpg.de>",
    "3.1",
    "Nenad Ocelic & David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, distance)

G_DEFINE_TYPE(GwyToolDistance, gwy_tool_distance, GWY_TYPE_PLAIN_TOOL)

static gboolean
module_register(void)
{
    gwy_tool_func_register(GWY_TYPE_TOOL_DISTANCE);

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, "distance");
    gwy_param_def_add_report_type(paramdef, PARAM_REPORT_STYLE, "report_style", _("Save Distance Table"),
                                  GWY_RESULTS_EXPORT_TABULAR_DATA, GWY_RESULTS_REPORT_TABSEP);
    /* TRANSLATORS: Number is verb here. */
    gwy_param_def_add_boolean(paramdef, PARAM_NUMBER_LINES, "number_lines", _("_Number lines"), TRUE);
    gwy_param_def_add_hold_selection(paramdef, PARAM_HOLD_SELECTION, "hold_selection", NULL);

    return paramdef;
}

static void
gwy_tool_distance_class_init(GwyToolDistanceClass *klass)
{
    GwyPlainToolClass *ptool_class = GWY_PLAIN_TOOL_CLASS(klass);
    GwyToolClass *tool_class = GWY_TOOL_CLASS(klass);
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->finalize = gwy_tool_distance_finalize;

    tool_class->icon_name = GWY_ICON_DISTANCE;
    tool_class->title = _("Distance");
    tool_class->tooltip = _("Measure distances and directions between points");
    tool_class->prefix = "/module/distance";
    tool_class->default_height = 240;
    tool_class->data_switched = gwy_tool_distance_data_switched;

    ptool_class->data_changed = gwy_tool_distance_data_changed;
    ptool_class->selection_changed = gwy_tool_distance_selection_changed;
}

static void
gwy_tool_distance_finalize(GObject *object)
{
    GwyToolDistance *tool = GWY_TOOL_DISTANCE(object);

    gwy_params_save_to_settings(tool->params);
    g_clear_object(&tool->params);
    if (tool->model) {
        gtk_tree_view_set_model(tool->treeview, NULL);
        g_clear_object(&tool->model);
    }
    GWY_FREE_VALUE_FORMAT(tool->angle_format);

    G_OBJECT_CLASS(gwy_tool_distance_parent_class)->finalize(object);
}

static void
gwy_tool_distance_init(GwyToolDistance *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
    plain_tool->unit_style = GWY_UNIT_FORMAT_MARKUP;
    plain_tool->lazy_updates = TRUE;

    tool->params = gwy_params_new_from_settings(define_module_params());
    tool->angle_format = gwy_value_format_new(1.0, 1, _("deg"));
    gwy_plain_tool_connect_selection(plain_tool, GWY_TYPE_LAYER_LINE, "line");
    gwy_plain_tool_enable_selection_holding(plain_tool);

    gwy_tool_distance_init_dialog(tool);
}

static void
gwy_tool_distance_init_dialog(GwyToolDistance *tool)
{
    GtkDialog *dialog = GTK_DIALOG(GWY_TOOL(tool)->dialog);
    GtkTreeViewColumn *column;
    GtkCellRenderer *renderer;
    GtkWidget *scwin, *label;
    GwyParamTable *table;
    GwyNullStore *store;
    guint i;

    store = gwy_null_store_new(0);
    tool->model = GTK_TREE_MODEL(store);
    tool->treeview = GTK_TREE_VIEW(gtk_tree_view_new_with_model(tool->model));
    gwy_plain_tool_enable_object_deletion(GWY_PLAIN_TOOL(tool), tool->treeview);

    for (i = 0; i < NCOLUMNS; i++) {
        column = gtk_tree_view_column_new();
        gtk_tree_view_column_set_expand(column, TRUE);
        gtk_tree_view_column_set_alignment(column, 0.5);
        g_object_set_data(G_OBJECT(column), "id", GUINT_TO_POINTER(i));
        renderer = gtk_cell_renderer_text_new();
        g_object_set(renderer, "xalign", 1.0, NULL);
        gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, TRUE);
        gtk_cell_layout_set_cell_data_func(GTK_CELL_LAYOUT(column), renderer, render_cell, tool, NULL);
        label = gtk_label_new(NULL);
        gtk_tree_view_column_set_widget(column, label);
        gtk_widget_show(label);
        gtk_tree_view_append_column(tool->treeview, column);
    }

    scwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(scwin), GTK_WIDGET(tool->treeview));
    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(dialog)), scwin, TRUE, TRUE, 0);

    table = tool->table = gwy_param_table_new(tool->params);
    gwy_param_table_append_report(table, PARAM_REPORT_STYLE);
    gwy_param_table_report_set_formatter(table, PARAM_REPORT_STYLE, format_report, tool, NULL);
    gwy_param_table_append_checkbox(table, PARAM_NUMBER_LINES);
    gwy_param_table_append_hold_selection(table, PARAM_HOLD_SELECTION);
    gwy_plain_tool_add_param_table(GWY_PLAIN_TOOL(tool), table);
    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(dialog)), gwy_param_table_widget(table), FALSE, TRUE, 0);

    gwy_plain_tool_add_clear_button(GWY_PLAIN_TOOL(tool));
    gwy_tool_add_hide_button(GWY_TOOL(tool), TRUE);
    gwy_help_add_to_tool_dialog(dialog, GWY_TOOL(tool), GWY_HELP_DEFAULT);

    update_headers(tool);

    g_signal_connect_swapped(tool->table, "param-changed", G_CALLBACK(param_changed), tool);

    gtk_widget_show_all(gtk_dialog_get_content_area(dialog));
}

static void
gwy_tool_distance_data_switched(GwyTool *gwytool, GwyDataView *data_view)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(gwytool);
    GwyToolDistance *tool = GWY_TOOL_DISTANCE(gwytool);
    gboolean ignore = (data_view == plain_tool->data_view);

    GWY_TOOL_CLASS(gwy_tool_distance_parent_class)->data_switched(gwytool, data_view);

    if (ignore || plain_tool->init_failed)
        return;

    if (data_view) {
        gwy_object_set_or_reset(plain_tool->layer, GWY_TYPE_LAYER_LINE,
                                "numbered", gwy_params_get_boolean(tool->params, PARAM_NUMBER_LINES),
                                "editable", TRUE,
                                "focus", -1,
                                NULL);
        gwy_selection_set_max_objects(plain_tool->selection, 1024);
        gwy_plain_tool_hold_selection(plain_tool, gwy_params_get_flags(tool->params, PARAM_HOLD_SELECTION));
    }
    update_headers(tool);
}

static void
gwy_tool_distance_data_changed(GwyPlainTool *plain_tool)
{
    update_headers(GWY_TOOL_DISTANCE(plain_tool));
}

static void
gwy_tool_distance_selection_changed(GwyPlainTool *plain_tool,
                                    gint hint)
{
    GwyToolDistance *tool = GWY_TOOL_DISTANCE(plain_tool);
    GwyNullStore *store = GWY_NULL_STORE(tool->model);
    gint n = gwy_null_store_get_n_rows(store);
    gboolean ok;

    g_return_if_fail(hint <= n);

    if (hint < 0) {
        gtk_tree_view_set_model(tool->treeview, NULL);
        n = (plain_tool->selection ? gwy_selection_get_data(plain_tool->selection, NULL) : 0);
        gwy_null_store_set_n_rows(store, n);
        gtk_tree_view_set_model(tool->treeview, tool->model);
    }
    else {
        GtkTreeSelection *selection;
        GtkTreePath *path;
        GtkTreeIter iter;

        if (hint < n)
            gwy_null_store_row_changed(store, hint);
        else
            gwy_null_store_set_n_rows(store, n+1);

        gtk_tree_model_iter_nth_child(tool->model, &iter, NULL, hint);
        path = gtk_tree_model_get_path(tool->model, &iter);
        selection = gtk_tree_view_get_selection(tool->treeview);
        gtk_tree_selection_select_iter(selection, &iter);
        gtk_tree_view_scroll_to_cell(tool->treeview, path, NULL, FALSE, 0.0, 0.0);
        gtk_tree_path_free(path);
    }

    ok = (plain_tool->selection && gwy_selection_get_data(plain_tool->selection, NULL));
    gwy_param_table_set_sensitive(tool->table, PARAM_REPORT_STYLE, ok);
}

static void
param_changed(GwyToolDistance *tool, gint id)
{
    if (id < 0 || id == PARAM_NUMBER_LINES) {
        GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
        GwyLayerLine *layer = plain_tool->layer ? GWY_LAYER_LINE(plain_tool->layer) : NULL;
        if (layer)
            gwy_layer_line_set_numbered(layer, gwy_params_get_boolean(tool->params, PARAM_NUMBER_LINES));
    }
}

static void
gwy_tool_distance_update_header(GwyToolDistance *tool,
                                guint col,
                                GString *str,
                                const gchar *title,
                                GwyValueFormat *vf)
{
    GtkTreeViewColumn *column = gtk_tree_view_get_column(tool->treeview, col);
    GtkLabel *label = GTK_LABEL(gtk_tree_view_column_get_widget(column));

    g_string_assign(str, "<b>");
    g_string_append(str, title);
    g_string_append(str, "</b>");
    if (vf)
        g_string_append_printf(str, " [%s]", vf->units);
    gtk_label_set_markup(label, str->str);
}

static void
update_headers(GwyToolDistance *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
    gboolean ok;
    GString *str;

    str = g_string_new(NULL);
    gwy_tool_distance_update_header(tool, COLUMN_I, str, "n", NULL);
    gwy_tool_distance_update_header(tool, COLUMN_DX, str, "Δx", plain_tool->coord_format);
    gwy_tool_distance_update_header(tool, COLUMN_DY, str, "Δy", plain_tool->coord_format);
    gwy_tool_distance_update_header(tool, COLUMN_PHI, str, "φ", tool->angle_format);
    gwy_tool_distance_update_header(tool, COLUMN_R, str, "R", plain_tool->coord_format);
    gwy_tool_distance_update_header(tool, COLUMN_DZ, str, "Δz", plain_tool->value_format);
    g_string_free(str, TRUE);

    ok = (plain_tool->selection && gwy_selection_get_data(plain_tool->selection, NULL));
    gwy_param_table_set_sensitive(tool->table, PARAM_REPORT_STYLE, ok);
}

static void
render_cell(GtkCellLayout *layout,
            GtkCellRenderer *renderer,
            GtkTreeModel *model,
            GtkTreeIter *iter,
            gpointer user_data)
{
    GwyToolDistance *tool = (GwyToolDistance*)user_data;
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
    gint id = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(layout), "id"));
    const GwyValueFormat *vf = NULL;
    gchar buf[32];
    gdouble line[4];
    gdouble val;
    guint idx;
    gint x, y;

    gtk_tree_model_get(model, iter, 0, &idx, -1);
    if (id == COLUMN_I) {
        g_snprintf(buf, sizeof(buf), "%d", idx + 1);
        g_object_set(renderer, "text", buf, NULL);
        return;
    }

    gwy_selection_get_object(plain_tool->selection, idx, line);

    if (id == COLUMN_DX) {
        vf = plain_tool->coord_format;
        val = line[2] - line[0];
    }
    else if (id == COLUMN_DY) {
        vf = plain_tool->coord_format;
        val = line[3] - line[1];
    }
    else if (id == COLUMN_R) {
        vf = plain_tool->coord_format;
        val = hypot(line[2] - line[0], line[3] - line[1]);
    }
    else if (id == COLUMN_PHI) {
        vf = tool->angle_format;
        val = gwy_rad2deg(atan2(line[1] - line[3], line[2] - line[0]));
    }
    else if (id == COLUMN_DZ) {
        x = floor(gwy_field_rtoj(plain_tool->field, line[2]));
        y = floor(gwy_field_rtoi(plain_tool->field, line[3]));
        val = gwy_field_get_val(plain_tool->field, x, y);
        x = floor(gwy_field_rtoj(plain_tool->field, line[0]));
        y = floor(gwy_field_rtoi(plain_tool->field, line[1]));
        val -= gwy_field_get_val(plain_tool->field, x, y);
        vf = plain_tool->value_format;
    }
    else {
        g_return_if_reached();
    }

    if (vf)
        g_snprintf(buf, sizeof(buf), "%.*f", vf->precision, val/vf->magnitude);
    else
        g_snprintf(buf, sizeof(buf), "%.3g", val);
    g_object_set(renderer, "text", buf, NULL);
}

static gchar*
format_report(gpointer user_data)
{
    GwyToolDistance *tool = (GwyToolDistance*)user_data;
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
    GwyResultsReportType report_style = gwy_params_get_report_type(tool->params, PARAM_REPORT_STYLE);
    GwyUnitFormatStyle style = GWY_UNIT_FORMAT_UNICODE;
    gint n, x, y, i;
    gdouble line[4], dx, dy, r, phi, dz;
    GwyValueFormat *vf_dist, *vf_phi, *vf_dz;
    GwyUnit *xyunit, *zunit;
    GString *text;
    gchar *dx_header, *dy_header, *phi_header, *r_header, *dz_header;

    text = g_string_new(NULL);
    xyunit = gwy_field_get_unit_xy(plain_tool->field);
    zunit = gwy_field_get_unit_z(plain_tool->field);
    if (!(report_style & GWY_RESULTS_REPORT_MACHINE)) {
        dx = gwy_field_get_dx(plain_tool->field);
        dy = gwy_field_get_dy(plain_tool->field);
        vf_dist = gwy_unit_get_format(xyunit, style, fmin(dx, dy), NULL);
        gwy_field_get_min_max(plain_tool->field, &dx, &dy);
        vf_dz = gwy_unit_get_format(zunit, style, fmax(fabs(dx), fabs(dy))/120.0, NULL);
        vf_phi = gwy_value_format_new(G_PI/180.0, 0, _("deg"));
    }
    else {
        vf_dist = gwy_unit_get_format_for_power10(xyunit, style, 0, NULL);
        vf_dz = gwy_unit_get_format_for_power10(zunit, style, 0, NULL);
        vf_phi = gwy_value_format_new(1.0, 0, "");
    }

    /* FIXME: This needs some helper function. */
    dx_header = g_strdup_printf("Δx [%s]", vf_dist->units);
    dy_header = g_strdup_printf("Δy [%s]", vf_dist->units);
    phi_header = g_strdup_printf("φ [%s]", vf_phi->units);
    r_header = g_strdup_printf("R [%s]", vf_dist->units);
    dz_header = g_strdup_printf("Δz [%s]", vf_dz->units);
    gwy_format_result_table_strings(text, report_style, 5, dx_header, dy_header, phi_header, r_header, dz_header);
    g_free(dx_header);
    g_free(dy_header);
    g_free(phi_header);
    g_free(r_header);
    g_free(dz_header);

    n = gwy_selection_get_data(plain_tool->selection, NULL);
    for (i = 0; i < n; i++) {
        gwy_selection_get_object(plain_tool->selection, i, line);

        dx = line[2] - line[0];
        dy = line[3] - line[1];
        r = hypot(line[2] - line[0], line[3] - line[1]);
        phi = atan2(line[1] - line[3], line[2] - line[0]);

        x = floor(gwy_field_rtoj(plain_tool->field, line[2]));
        y = floor(gwy_field_rtoi(plain_tool->field, line[3]));
        dz = gwy_field_get_val(plain_tool->field, x, y);
        x = floor(gwy_field_rtoj(plain_tool->field, line[0]));
        y = floor(gwy_field_rtoi(plain_tool->field, line[1]));
        dz -= gwy_field_get_val(plain_tool->field, x, y);

        gwy_format_result_table_row(text, report_style, 5,
                                    dx/vf_dist->magnitude, dy/vf_dist->magnitude,
                                    phi/vf_phi->magnitude, r/vf_dist->magnitude, dz/vf_dz->magnitude);
    }

    return g_string_free(text, FALSE);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
