/*  This file is part of SAIL (https://github.com/HappySeaFox/sail)

    Copyright (c) 2020-2021 Dmitry Baryshev

    The MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
*/

#include <limits.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>

#include <sail-manip/sail-manip.h>

#include "fast_conversions.h"

/*
 * Private functions.
 */

/* Helper functions for float<->uint16 conversion */
static inline uint16_t float_to_uint16(float value)
{
    /* Clamp to [0.0, 1.0] and scale to [0, 65535] */
    if (value <= 0.0f) return 0;
    if (value >= 1.0f) return 65535;
    return (uint16_t)(value * 65535.0f + 0.5f);
}

static inline float uint16_to_float(uint16_t value)
{
    /* Scale [0, 65535] to [0.0, 1.0] */
    return (float)value / 65535.0f;
}

/* Half-precision (IEEE 754 binary16) conversion helpers */
static inline uint16_t float32_to_float16(float value)
{
    union { float f; uint32_t i; } v = { .f = value };
    uint32_t i = v.i;

    int32_t sign = (i >> 16) & 0x8000;
    int32_t exponent = ((i >> 23) & 0xff) - 127 + 15;
    int32_t mantissa = i & 0x007fffff;

    if (exponent <= 0)
    {
        if (exponent < -10)
        {
            return (uint16_t)sign; /* Too small, return signed zero */
        }
        mantissa = (mantissa | 0x00800000) >> (1 - exponent);
        return (uint16_t)(sign | (mantissa >> 13));
    }
    else if (exponent == 0xff - 127 + 15)
    {
        if (mantissa == 0)
        {
            return (uint16_t)(sign | 0x7c00); /* Infinity */
        }
        else
        {
            return (uint16_t)(sign | 0x7c00 | (mantissa >> 13)); /* NaN */
        }
    }
    else if (exponent > 30)
    {
        return (uint16_t)(sign | 0x7c00); /* Overflow to infinity */
    }

    return (uint16_t)(sign | (exponent << 10) | (mantissa >> 13));
}

static inline float float16_to_float32(uint16_t value)
{
    uint32_t sign = (uint32_t)(value & 0x8000) << 16;
    uint32_t exponent = (value >> 10) & 0x1f;
    uint32_t mantissa = value & 0x3ff;

    if (exponent == 0)
    {
        if (mantissa == 0)
        {
            /* Zero */
            union { float f; uint32_t i; } v = { .i = sign };
            return v.f;
        }
        else
        {
            /* Denormalized */
            while (!(mantissa & 0x400))
            {
                mantissa <<= 1;
                exponent--;
            }
            exponent++;
            mantissa &= ~0x400;
        }
    }
    else if (exponent == 31)
    {
        /* Infinity or NaN */
        union { float f; uint32_t i; } v = { .i = sign | 0x7f800000 | (mantissa << 13) };
        return v.f;
    }

    exponent = exponent + (127 - 15);
    mantissa = mantissa << 13;

    union { float f; uint32_t i; } v = { .i = sign | (exponent << 23) | mantissa };
    return v.f;
}

static inline uint16_t half_to_uint16(uint16_t half_value)
{
    float f = float16_to_float32(half_value);

    if (f != f) /* NaN. */
    {
        return 0;
    }
    else if (f <= 0.0f)
    {
        return 0;
    }
    else if (f >= 1.0f)
    {
        return 65535;
    }
    else
    {
        return (uint16_t)(f * 65535.0f + 0.5f);
    }
}

static inline uint16_t uint16_to_half(uint16_t value)
{
    float f = uint16_to_float(value);
    return float32_to_float16(f);
}

struct output_context
{
    struct sail_image* image;
    int r;
    int g;
    int b;
    int a;
    const struct sail_conversion_options* options;
};

typedef void (*pixel_consumer_t)(const struct output_context* output_context,
                                 uint8_t** scan8,
                                 uint16_t** scan16,
                                 const sail_rgba32_t* rgba32,
                                 const sail_rgba64_t* rgba64);

static inline void pixel_consumer_gray8(const struct output_context* output_context,
                                        uint8_t** scan8,
                                        uint16_t** scan16,
                                        const sail_rgba32_t* rgba32,
                                        const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_gray8_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_gray8_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    (*scan8)++;
}

static inline void pixel_consumer_gray16(const struct output_context* output_context,
                                         uint8_t** scan8,
                                         uint16_t** scan16,
                                         const sail_rgba32_t* rgba32,
                                         const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_gray16_pixel_from_uint8_values(rgba32, *scan16, output_context->options);
    }
    else
    {
        fill_gray16_pixel_from_uint16_values(rgba64, *scan16, output_context->options);
    }

    (*scan16)++;
}

static inline void pixel_consumer_rgb24_kind(const struct output_context* output_context,
                                             uint8_t** scan8,
                                             uint16_t** scan16,
                                             const sail_rgba32_t* rgba32,
                                             const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_rgb24_pixel_from_uint8_values(rgba32, *scan8, output_context->r, output_context->g, output_context->b,
                                           output_context->options);
    }
    else
    {
        fill_rgb24_pixel_from_uint16_values(rgba64, *scan8, output_context->r, output_context->g, output_context->b,
                                            output_context->options);
    }

    *scan8 += 3;
}

static inline void pixel_consumer_rgb48_kind(const struct output_context* output_context,
                                             uint8_t** scan8,
                                             uint16_t** scan16,
                                             const sail_rgba32_t* rgba32,
                                             const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_rgb48_pixel_from_uint8_values(rgba32, *scan16, output_context->r, output_context->g, output_context->b,
                                           output_context->options);
    }
    else
    {
        fill_rgb48_pixel_from_uint16_values(rgba64, *scan16, output_context->r, output_context->g, output_context->b,
                                            output_context->options);
    }

    *scan16 += 3;
}

static inline void pixel_consumer_rgba32_kind(const struct output_context* output_context,
                                              uint8_t** scan8,
                                              uint16_t** scan16,
                                              const sail_rgba32_t* rgba32,
                                              const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_rgba32_pixel_from_uint8_values(rgba32, *scan8, output_context->r, output_context->g, output_context->b,
                                            output_context->a, output_context->options);
    }
    else
    {
        fill_rgba32_pixel_from_uint16_values(rgba64, *scan8, output_context->r, output_context->g, output_context->b,
                                             output_context->a, output_context->options);
    }

    *scan8 += 4;
}

static inline void pixel_consumer_rgba64_kind(const struct output_context* output_context,
                                              uint8_t** scan8,
                                              uint16_t** scan16,
                                              const sail_rgba32_t* rgba32,
                                              const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_rgba64_pixel_from_uint8_values(rgba32, *scan16, output_context->r, output_context->g, output_context->b,
                                            output_context->a, output_context->options);
    }
    else
    {
        fill_rgba64_pixel_from_uint16_values(rgba64, *scan16, output_context->r, output_context->g, output_context->b,
                                             output_context->a, output_context->options);
    }

    *scan16 += 4;
}

static inline void pixel_consumer_ycbcr(const struct output_context* output_context,
                                        uint8_t** scan8,
                                        uint16_t** scan16,
                                        const sail_rgba32_t* rgba32,
                                        const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_ycbcr_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_ycbcr_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    *scan8 += 3;
}

static inline void pixel_consumer_gray_alpha8(const struct output_context* output_context,
                                              uint8_t** scan8,
                                              uint16_t** scan16,
                                              const sail_rgba32_t* rgba32,
                                              const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_gray_alpha8_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_gray_alpha8_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    (*scan8)++;
}

static inline void pixel_consumer_gray_alpha16(const struct output_context* output_context,
                                               uint8_t** scan8,
                                               uint16_t** scan16,
                                               const sail_rgba32_t* rgba32,
                                               const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_gray_alpha16_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_gray_alpha16_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    *scan8 += 2;
}

static inline void pixel_consumer_gray_alpha32(const struct output_context* output_context,
                                               uint8_t** scan8,
                                               uint16_t** scan16,
                                               const sail_rgba32_t* rgba32,
                                               const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_gray_alpha32_pixel_from_uint8_values(rgba32, *scan16, output_context->options);
    }
    else
    {
        fill_gray_alpha32_pixel_from_uint16_values(rgba64, *scan16, output_context->options);
    }

    *scan16 += 2;
}

static inline void pixel_consumer_rgb555_kind(const struct output_context* output_context,
                                              uint8_t** scan8,
                                              uint16_t** scan16,
                                              const sail_rgba32_t* rgba32,
                                              const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_rgb555_pixel_from_uint8_values(rgba32, *scan16, output_context->r, output_context->g, output_context->b,
                                            output_context->options);
    }
    else
    {
        fill_rgb555_pixel_from_uint16_values(rgba64, *scan16, output_context->r, output_context->g, output_context->b,
                                             output_context->options);
    }

    (*scan16)++;
}

static inline void pixel_consumer_rgb565_kind(const struct output_context* output_context,
                                              uint8_t** scan8,
                                              uint16_t** scan16,
                                              const sail_rgba32_t* rgba32,
                                              const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_rgb565_pixel_from_uint8_values(rgba32, *scan16, output_context->r, output_context->g, output_context->b,
                                            output_context->options);
    }
    else
    {
        fill_rgb565_pixel_from_uint16_values(rgba64, *scan16, output_context->r, output_context->g, output_context->b,
                                             output_context->options);
    }

    (*scan16)++;
}

static inline void pixel_consumer_cmyk32(const struct output_context* output_context,
                                         uint8_t** scan8,
                                         uint16_t** scan16,
                                         const sail_rgba32_t* rgba32,
                                         const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_cmyk32_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_cmyk32_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    *scan8 += 4;
}

static inline void pixel_consumer_cmyk64(const struct output_context* output_context,
                                         uint8_t** scan8,
                                         uint16_t** scan16,
                                         const sail_rgba32_t* rgba32,
                                         const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_cmyk64_pixel_from_uint8_values(rgba32, *scan16, output_context->options);
    }
    else
    {
        fill_cmyk64_pixel_from_uint16_values(rgba64, *scan16, output_context->options);
    }

    *scan16 += 4;
}

static inline void pixel_consumer_cmyka40(const struct output_context* output_context,
                                          uint8_t** scan8,
                                          uint16_t** scan16,
                                          const sail_rgba32_t* rgba32,
                                          const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_cmyka40_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_cmyka40_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    *scan8 += 5;
}

static inline void pixel_consumer_cmyka80(const struct output_context* output_context,
                                          uint8_t** scan8,
                                          uint16_t** scan16,
                                          const sail_rgba32_t* rgba32,
                                          const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_cmyka80_pixel_from_uint8_values(rgba32, *scan16, output_context->options);
    }
    else
    {
        fill_cmyka80_pixel_from_uint16_values(rgba64, *scan16, output_context->options);
    }

    *scan16 += 5;
}

static inline void pixel_consumer_rgba16_kind(const struct output_context* output_context,
                                              uint8_t** scan8,
                                              uint16_t** scan16,
                                              const sail_rgba32_t* rgba32,
                                              const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        fill_rgba16_pixel_from_uint8_values(rgba32, *scan16, output_context->r, output_context->g, output_context->b,
                                            output_context->a, 4, output_context->options);
    }
    else
    {
        fill_rgba16_pixel_from_uint16_values(rgba64, *scan16, output_context->r, output_context->g, output_context->b,
                                             output_context->a, 4, output_context->options);
    }

    (*scan16)++;
}

static inline void pixel_consumer_yuv24(const struct output_context* output_context,
                                        uint8_t** scan8,
                                        uint16_t** scan16,
                                        const sail_rgba32_t* rgba32,
                                        const sail_rgba64_t* rgba64)
{
    (void)scan16;

    if (rgba32 != NULL)
    {
        fill_yuv24_pixel_from_uint8_values(rgba32, *scan8, output_context->options);
    }
    else
    {
        fill_yuv24_pixel_from_uint16_values(rgba64, *scan8, output_context->options);
    }

    *scan8 += 3;
}

/* Floating point pixel consumers */
static inline void pixel_consumer_gray_half(const struct output_context* output_context,
                                            uint8_t** scan8,
                                            uint16_t** scan16,
                                            const sail_rgba32_t* rgba32,
                                            const sail_rgba64_t* rgba64)
{
    (void)scan8;

    uint16_t gray;
    if (rgba32 != NULL)
    {
        fill_gray16_pixel_from_uint8_values(rgba32, &gray, output_context->options);
    }
    else
    {
        fill_gray16_pixel_from_uint16_values(rgba64, &gray, output_context->options);
    }

    **scan16 = uint16_to_half(gray);
    (*scan16)++;
}

static inline void pixel_consumer_gray_float(const struct output_context* output_context,
                                             uint8_t** scan8,
                                             uint16_t** scan16,
                                             const sail_rgba32_t* rgba32,
                                             const sail_rgba64_t* rgba64)
{
    (void)scan16;

    uint16_t gray;
    if (rgba32 != NULL)
    {
        fill_gray16_pixel_from_uint8_values(rgba32, &gray, output_context->options);
    }
    else
    {
        fill_gray16_pixel_from_uint16_values(rgba64, &gray, output_context->options);
    }

    float* scan_float = (float*)*scan8;
    *scan_float = uint16_to_float(gray);
    *scan8 = (uint8_t*)(scan_float + 1);
}

static inline void pixel_consumer_rgb_half(const struct output_context* output_context,
                                           uint8_t** scan8,
                                           uint16_t** scan16,
                                           const sail_rgba32_t* rgba32,
                                           const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        *((*scan16) + output_context->r) = uint16_to_half((uint16_t)(rgba32->component1 * 257));
        *((*scan16) + output_context->g) = uint16_to_half((uint16_t)(rgba32->component2 * 257));
        *((*scan16) + output_context->b) = uint16_to_half((uint16_t)(rgba32->component3 * 257));
    }
    else
    {
        *((*scan16) + output_context->r) = uint16_to_half(rgba64->component1);
        *((*scan16) + output_context->g) = uint16_to_half(rgba64->component2);
        *((*scan16) + output_context->b) = uint16_to_half(rgba64->component3);
    }

    *scan16 += 3;
}

static inline void pixel_consumer_rgba_half(const struct output_context* output_context,
                                            uint8_t** scan8,
                                            uint16_t** scan16,
                                            const sail_rgba32_t* rgba32,
                                            const sail_rgba64_t* rgba64)
{
    (void)scan8;

    if (rgba32 != NULL)
    {
        *((*scan16) + output_context->r) = uint16_to_half((uint16_t)(rgba32->component1 * 257));
        *((*scan16) + output_context->g) = uint16_to_half((uint16_t)(rgba32->component2 * 257));
        *((*scan16) + output_context->b) = uint16_to_half((uint16_t)(rgba32->component3 * 257));
        *((*scan16) + output_context->a) = uint16_to_half((uint16_t)(rgba32->component4 * 257));
    }
    else
    {
        *((*scan16) + output_context->r) = uint16_to_half(rgba64->component1);
        *((*scan16) + output_context->g) = uint16_to_half(rgba64->component2);
        *((*scan16) + output_context->b) = uint16_to_half(rgba64->component3);
        *((*scan16) + output_context->a) = uint16_to_half(rgba64->component4);
    }

    *scan16 += 4;
}

static inline void pixel_consumer_rgb_float(const struct output_context* output_context,
                                            uint8_t** scan8,
                                            uint16_t** scan16,
                                            const sail_rgba32_t* rgba32,
                                            const sail_rgba64_t* rgba64)
{
    (void)scan16;

    float* scan_float = (float*)*scan8;

    if (rgba32 != NULL)
    {
        *(scan_float + output_context->r) = uint16_to_float((uint16_t)(rgba32->component1 * 257));
        *(scan_float + output_context->g) = uint16_to_float((uint16_t)(rgba32->component2 * 257));
        *(scan_float + output_context->b) = uint16_to_float((uint16_t)(rgba32->component3 * 257));
    }
    else
    {
        *(scan_float + output_context->r) = uint16_to_float(rgba64->component1);
        *(scan_float + output_context->g) = uint16_to_float(rgba64->component2);
        *(scan_float + output_context->b) = uint16_to_float(rgba64->component3);
    }

    *scan8 = (uint8_t*)(scan_float + 3);
}

static inline void pixel_consumer_rgba_float(const struct output_context* output_context,
                                             uint8_t** scan8,
                                             uint16_t** scan16,
                                             const sail_rgba32_t* rgba32,
                                             const sail_rgba64_t* rgba64)
{
    (void)scan16;

    float* scan_float = (float*)*scan8;

    if (rgba32 != NULL)
    {
        *(scan_float + output_context->r) = uint16_to_float((uint16_t)(rgba32->component1 * 257));
        *(scan_float + output_context->g) = uint16_to_float((uint16_t)(rgba32->component2 * 257));
        *(scan_float + output_context->b) = uint16_to_float((uint16_t)(rgba32->component3 * 257));
        *(scan_float + output_context->a) = uint16_to_float((uint16_t)(rgba32->component4 * 257));
    }
    else
    {
        *(scan_float + output_context->r) = uint16_to_float(rgba64->component1);
        *(scan_float + output_context->g) = uint16_to_float(rgba64->component2);
        *(scan_float + output_context->b) = uint16_to_float(rgba64->component3);
        *(scan_float + output_context->a) = uint16_to_float(rgba64->component4);
    }

    *scan8 = (uint8_t*)(scan_float + 4);
}

static bool verify_and_construct_rgba_indexes_silent(
    enum SailPixelFormat output_pixel_format, pixel_consumer_t* pixel_consumer, int* r, int* g, int* b, int* a)
{
    switch (output_pixel_format)
    {
    case SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE:
    {
        *pixel_consumer = pixel_consumer_gray8;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE:
    {
        *pixel_consumer = pixel_consumer_gray16;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE_ALPHA:
    {
        *pixel_consumer = pixel_consumer_gray_alpha8;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA:
    {
        *pixel_consumer = pixel_consumer_gray_alpha16;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA:
    {
        *pixel_consumer = pixel_consumer_gray_alpha32;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP16_RGB555:
    {
        *pixel_consumer = pixel_consumer_rgb555_kind;
        *r              = 0;
        *g              = 5;
        *b              = 10;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_BGR555:
    {
        *pixel_consumer = pixel_consumer_rgb555_kind;
        *r              = 10;
        *g              = 5;
        *b              = 0;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_RGB565:
    {
        *pixel_consumer = pixel_consumer_rgb565_kind;
        *r              = 0;
        *g              = 5;
        *b              = 11;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_BGR565:
    {
        *pixel_consumer = pixel_consumer_rgb565_kind;
        *r              = 11;
        *g              = 5;
        *b              = 0;
        *a              = -1;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP24_RGB:
    {
        *pixel_consumer = pixel_consumer_rgb24_kind;
        *r              = 0;
        *g              = 1;
        *b              = 2;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP24_BGR:
    {
        *pixel_consumer = pixel_consumer_rgb24_kind;
        *r              = 2;
        *g              = 1;
        *b              = 0;
        *a              = -1;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP48_RGB:
    {
        *pixel_consumer = pixel_consumer_rgb48_kind;
        *r              = 0;
        *g              = 1;
        *b              = 2;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP48_BGR:
    {
        *pixel_consumer = pixel_consumer_rgb48_kind;
        *r              = 2;
        *g              = 1;
        *b              = 0;
        *a              = -1;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP16_RGBX:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 0;
        *g              = 4;
        *b              = 8;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_BGRX:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 8;
        *g              = 4;
        *b              = 0;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_XRGB:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 4;
        *g              = 8;
        *b              = 12;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_XBGR:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 12;
        *g              = 8;
        *b              = 4;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_RGBA:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 0;
        *g              = 4;
        *b              = 8;
        *a              = 12;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_BGRA:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 8;
        *g              = 4;
        *b              = 0;
        *a              = 12;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_ARGB:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 4;
        *g              = 8;
        *b              = 12;
        *a              = 0;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_ABGR:
    {
        *pixel_consumer = pixel_consumer_rgba16_kind;
        *r              = 12;
        *g              = 8;
        *b              = 4;
        *a              = 0;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP32_RGBX:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 0;
        *g              = 1;
        *b              = 2;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_BGRX:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 2;
        *g              = 1;
        *b              = 0;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_XRGB:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 1;
        *g              = 2;
        *b              = 3;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_XBGR:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 3;
        *g              = 2;
        *b              = 1;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_RGBA:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 0;
        *g              = 1;
        *b              = 2;
        *a              = 3;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_BGRA:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 2;
        *g              = 1;
        *b              = 0;
        *a              = 3;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_ARGB:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 1;
        *g              = 2;
        *b              = 3;
        *a              = 0;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_ABGR:
    {
        *pixel_consumer = pixel_consumer_rgba32_kind;
        *r              = 3;
        *g              = 2;
        *b              = 1;
        *a              = 0;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP64_RGBX:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 0;
        *g              = 1;
        *b              = 2;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_BGRX:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 2;
        *g              = 1;
        *b              = 0;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_XRGB:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 1;
        *g              = 2;
        *b              = 3;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_XBGR:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 3;
        *g              = 2;
        *b              = 1;
        *a              = -1;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_RGBA:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 0;
        *g              = 1;
        *b              = 2;
        *a              = 3;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_BGRA:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 2;
        *g              = 1;
        *b              = 0;
        *a              = 3;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_ARGB:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 1;
        *g              = 2;
        *b              = 3;
        *a              = 0;
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_ABGR:
    {
        *pixel_consumer = pixel_consumer_rgba64_kind;
        *r              = 3;
        *g              = 2;
        *b              = 1;
        *a              = 0;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP32_CMYK:
    {
        *pixel_consumer = pixel_consumer_cmyk32;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_CMYK:
    {
        *pixel_consumer = pixel_consumer_cmyk64;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP40_CMYKA:
    {
        *pixel_consumer = pixel_consumer_cmyka40;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP80_CMYKA:
    {
        *pixel_consumer = pixel_consumer_cmyka80;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP24_YCBCR:
    {
        *pixel_consumer = pixel_consumer_ycbcr;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP24_YUV:
    {
        *pixel_consumer = pixel_consumer_yuv24;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_HALF:
    {
        *pixel_consumer = pixel_consumer_gray_half;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_FLOAT:
    {
        *pixel_consumer = pixel_consumer_gray_float;
        *r = *g = *b = *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP48_RGB_HALF:
    {
        *pixel_consumer = pixel_consumer_rgb_half;
        *r = 0;
        *g = 1;
        *b = 2;
        *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP64_RGBA_HALF:
    {
        *pixel_consumer = pixel_consumer_rgba_half;
        *r = 0;
        *g = 1;
        *b = 2;
        *a = 3;
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT:
    {
        *pixel_consumer = pixel_consumer_rgb_float;
        *r = 0;
        *g = 1;
        *b = 2;
        *a = -1; /* unused. */
        break;
    }

    case SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT:
    {
        *pixel_consumer = pixel_consumer_rgba_float;
        *r = 0;
        *g = 1;
        *b = 2;
        *a = 3;
        break;
    }

    default:
    {
        return false;
    }
    }

    return true;
}

static sail_status_t verify_and_construct_rgba_indexes_verbose(
    enum SailPixelFormat output_pixel_format, pixel_consumer_t* pixel_consumer, int* r, int* g, int* b, int* a)
{
    if (verify_and_construct_rgba_indexes_silent(output_pixel_format, pixel_consumer, r, g, b, a))
    {
        return SAIL_OK;
    }
    else
    {
        SAIL_LOG_ERROR("Conversion to %s is not supported", sail_pixel_format_to_string(output_pixel_format));
        SAIL_LOG_AND_RETURN(SAIL_ERROR_UNSUPPORTED_PIXEL_FORMAT);
    }
}

static sail_status_t convert_from_indexed(const struct sail_image* image,
                                          const unsigned input_bit_shift,
                                          const unsigned bit_shift_decrease_by,
                                          const unsigned input_bit_mask,
                                          const unsigned bit_mask_shift_by,
                                          pixel_consumer_t pixel_consumer,
                                          const struct output_context* output_context)
{
    /* Pre-convert palette once to avoid repeated lookups and format conversions */
    sail_rgba32_t* preconverted_palette = NULL;
    SAIL_TRY(preconvert_palette_to_rgba32(image->palette, &preconverted_palette));

    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width;)
        {
            unsigned bit_shift = input_bit_shift;
            unsigned bit_mask  = input_bit_mask;
            const uint8_t byte = *scan_input++;

            while (bit_mask > 0 && column < image->width)
            {
                const uint8_t index = (byte & bit_mask) >> bit_shift;

                /* Direct lookup in pre-converted palette. */
                const uint8_t safe_index   = (index < image->palette->color_count) ? index : 0;
                const sail_rgba32_t rgba32 = preconverted_palette[safe_index];
                pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);

                bit_shift  -= bit_shift_decrease_by;
                bit_mask  >>= bit_mask_shift_by;
                column++;
            }
        }
    }

    sail_free(preconverted_palette);
    return SAIL_OK;
}

static sail_status_t convert_from_bpp1_indexed(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    SAIL_TRY(convert_from_indexed(image, 7, 1, /* 10000000 */ 0x80, 1, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp2_indexed(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    SAIL_TRY(convert_from_indexed(image, 6, 2, /* 11000000 */ 0xC0, 2, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp4_indexed(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    SAIL_TRY(convert_from_indexed(image, 4, 4, /* 11110000 */ 0xF0, 4, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp8_indexed(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    SAIL_TRY(convert_from_indexed(image, 0, 0, /* 11111111 */ 0xFF, 8, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_grayscale_up_to_bpp8(const struct sail_image* image,
                                                       const unsigned input_bit_shift,
                                                       const unsigned bit_shift_decrease_by,
                                                       const unsigned input_bit_mask,
                                                       const unsigned bit_mask_shift_by,
                                                       const unsigned multiplicator_to_255,
                                                       pixel_consumer_t pixel_consumer,
                                                       const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width;)
        {
            unsigned bit_shift = input_bit_shift;
            unsigned bit_mask  = input_bit_mask;
            const uint8_t byte = *scan_input++;

            while (bit_mask > 0 && column < image->width)
            {
                const uint8_t index = (byte & bit_mask) >> bit_shift;

                sail_rgba32_t rgba32;
                spread_gray8_to_rgba32((uint8_t)(index * multiplicator_to_255), &rgba32);
                pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);

                bit_shift  -= bit_shift_decrease_by;
                bit_mask  >>= bit_mask_shift_by;
                column++;
            }
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp1_grayscale(const struct sail_image* image,
                                                 pixel_consumer_t pixel_consumer,
                                                 const struct output_context* output_context)
{
    SAIL_TRY(
        convert_from_grayscale_up_to_bpp8(image, 7, 1, /* 10000000 */ 0x80, 1, 255, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp2_grayscale(const struct sail_image* image,
                                                 pixel_consumer_t pixel_consumer,
                                                 const struct output_context* output_context)
{
    SAIL_TRY(
        convert_from_grayscale_up_to_bpp8(image, 6, 2, /* 11000000 */ 0xC0, 2, 85, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp4_grayscale(const struct sail_image* image,
                                                 pixel_consumer_t pixel_consumer,
                                                 const struct output_context* output_context)
{
    SAIL_TRY(
        convert_from_grayscale_up_to_bpp8(image, 4, 4, /* 11110000 */ 0xF0, 4, 17, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp8_grayscale(const struct sail_image* image,
                                                 pixel_consumer_t pixel_consumer,
                                                 const struct output_context* output_context)
{
    SAIL_TRY(convert_from_grayscale_up_to_bpp8(image, 0, 0, /* 11111111 */ 0xFF, 8, 1, pixel_consumer, output_context));

    return SAIL_OK;
}

static sail_status_t convert_from_bpp16_grayscale(const struct sail_image* image,
                                                  pixel_consumer_t pixel_consumer,
                                                  const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba64_t rgba64;
            spread_gray16_to_rgba64(*scan_input++, &rgba64);
            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp16_grayscale_alpha(const struct sail_image* image,
                                                        pixel_consumer_t pixel_consumer,
                                                        const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba32_t rgba32;
            spread_gray8_to_rgba32(*scan_input++, &rgba32);
            rgba32.component4 = *scan_input++;

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp32_grayscale_alpha(const struct sail_image* image,
                                                        pixel_consumer_t pixel_consumer,
                                                        const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba64_t rgba64;
            spread_gray16_to_rgba64(*scan_input++, &rgba64);
            rgba64.component4 = *scan_input++;

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp16_rgb555(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba32_t rgba32 = {((*scan_input >> 0) & 0x1f) << 3, ((*scan_input >> 5) & 0x1f) << 3,
                                          ((*scan_input >> 10) & 0x1f) << 3, 255};

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input++;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp16_bgr555(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba32_t rgba32 = {((*scan_input >> 10) & 0x1f) << 3, ((*scan_input >> 5) & 0x1f) << 3,
                                          ((*scan_input >> 0) & 0x1f) << 3, 255};

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input++;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp16_rgb565(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba32_t rgba32 = {((*scan_input >> 0) & 0x1f) << 3, ((*scan_input >> 5) & 0x3f) << 2,
                                          ((*scan_input >> 11) & 0x1f) << 3, 255};

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input++;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp16_bgr565(const struct sail_image* image,
                                               pixel_consumer_t pixel_consumer,
                                               const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba32_t rgba32 = {((*scan_input >> 11) & 0x1f) << 3, ((*scan_input >> 5) & 0x3f) << 2,
                                          ((*scan_input >> 0) & 0x1f) << 3, 255};

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input++;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp24_rgb_kind(const struct sail_image* image,
                                                 int ri,
                                                 int gi,
                                                 int bi,
                                                 pixel_consumer_t pixel_consumer,
                                                 const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba32_t rgba32 = {*(scan_input + ri), *(scan_input + gi), *(scan_input + bi), 255};

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input += 3;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp48_rgb_kind(const struct sail_image* image,
                                                 int ri,
                                                 int gi,
                                                 int bi,
                                                 pixel_consumer_t pixel_consumer,
                                                 const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba64_t rgba64 = {*(scan_input + ri), *(scan_input + gi), *(scan_input + bi), 65535};

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 3;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp32_rgba_kind(const struct sail_image* image,
                                                  int ri,
                                                  int gi,
                                                  int bi,
                                                  int ai,
                                                  pixel_consumer_t pixel_consumer,
                                                  const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba32_t rgba32 = {*(scan_input + ri), *(scan_input + gi), *(scan_input + bi),
                                          ai >= 0 ? *(scan_input + ai) : 255};

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp64_rgba_kind(const struct sail_image* image,
                                                  int ri,
                                                  int gi,
                                                  int bi,
                                                  int ai,
                                                  pixel_consumer_t pixel_consumer,
                                                  const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba64_t rgba64 = {*(scan_input + ri), *(scan_input + gi), *(scan_input + bi),
                                          ai >= 0 ? *(scan_input + ai) : 65535};

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp32_cmyk(const struct sail_image* image,
                                             pixel_consumer_t pixel_consumer,
                                             const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba32_t rgba32;
            convert_cmyk32_to_rgba32(*(scan_input + 0), *(scan_input + 1), *(scan_input + 2), *(scan_input + 3),
                                     &rgba32);

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp64_cmyk(const struct sail_image* image,
                                             pixel_consumer_t pixel_consumer,
                                             const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba64_t rgba64;
            convert_cmyk64_to_rgba64(*(scan_input + 0), *(scan_input + 1), *(scan_input + 2), *(scan_input + 3),
                                     &rgba64);

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp40_cmyka(const struct sail_image* image,
                                              pixel_consumer_t pixel_consumer,
                                              const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba32_t rgba32;
            convert_cmyka40_to_rgba32(*(scan_input + 0), *(scan_input + 1), *(scan_input + 2), *(scan_input + 3),
                                      *(scan_input + 4), &rgba32);

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input += 5;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp80_cmyka(const struct sail_image* image,
                                              pixel_consumer_t pixel_consumer,
                                              const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba64_t rgba64;
            convert_cmyka80_to_rgba64(*(scan_input + 0), *(scan_input + 1), *(scan_input + 2), *(scan_input + 3),
                                      *(scan_input + 4), &rgba64);

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 5;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp24_ycbcr(const struct sail_image* image,
                                              pixel_consumer_t pixel_consumer,
                                              const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba32_t rgba32;
            convert_ycbcr24_to_rgba32(*(scan_input + 0), *(scan_input + 1), *(scan_input + 2), &rgba32);

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input += 3;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp32_ycck(const struct sail_image* image,
                                             pixel_consumer_t pixel_consumer,
                                             const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint8_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8     = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16   = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba32_t rgba32;
            convert_ycck32_to_rgba32(*(scan_input + 0), *(scan_input + 1), *(scan_input + 2), *(scan_input + 3),
                                     &rgba32);

            pixel_consumer(output_context, &scan_output8, &scan_output16, &rgba32, NULL);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

/* Floating point format conversions */
static sail_status_t convert_from_bpp16_grayscale_half(const struct sail_image* image,
                                                        pixel_consumer_t pixel_consumer,
                                                        const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba64_t rgba64;
            uint16_t gray = half_to_uint16(*scan_input++);
            spread_gray16_to_rgba64(gray, &rgba64);
            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp32_grayscale_float(const struct sail_image* image,
                                                         pixel_consumer_t pixel_consumer,
                                                         const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const float* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8   = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16 = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            sail_rgba64_t rgba64;
            uint16_t gray = float_to_uint16(*scan_input++);
            spread_gray16_to_rgba64(gray, &rgba64);
            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp48_rgb_half(const struct sail_image* image,
                                                  pixel_consumer_t pixel_consumer,
                                                  const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba64_t rgba64 = {
                half_to_uint16(*(scan_input + 0)),
                half_to_uint16(*(scan_input + 1)),
                half_to_uint16(*(scan_input + 2)),
                65535
            };

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 3;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp64_rgba_half(const struct sail_image* image,
                                                   pixel_consumer_t pixel_consumer,
                                                   const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const uint16_t* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8      = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16    = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba64_t rgba64 = {
                half_to_uint16(*(scan_input + 0)),
                half_to_uint16(*(scan_input + 1)),
                half_to_uint16(*(scan_input + 2)),
                half_to_uint16(*(scan_input + 3))
            };

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp96_rgb_float(const struct sail_image* image,
                                                   pixel_consumer_t pixel_consumer,
                                                   const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const float* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8   = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16 = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba64_t rgba64 = {
                float_to_uint16(*(scan_input + 0)),
                float_to_uint16(*(scan_input + 1)),
                float_to_uint16(*(scan_input + 2)),
                65535
            };

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 3;
        }
    }

    return SAIL_OK;
}

static sail_status_t convert_from_bpp128_rgba_float(const struct sail_image* image,
                                                     pixel_consumer_t pixel_consumer,
                                                     const struct output_context* output_context)
{
    unsigned row;

#pragma omp parallel for schedule(SAIL_OPENMP_SCHEDULE)
    for (row = 0; row < image->height; row++)
    {
        const float* scan_input = sail_scan_line(image, row);
        uint8_t* scan_output8   = sail_scan_line(output_context->image, row);
        uint16_t* scan_output16 = sail_scan_line(output_context->image, row);

        for (unsigned column = 0; column < image->width; column++)
        {
            const sail_rgba64_t rgba64 = {
                float_to_uint16(*(scan_input + 0)),
                float_to_uint16(*(scan_input + 1)),
                float_to_uint16(*(scan_input + 2)),
                float_to_uint16(*(scan_input + 3))
            };

            pixel_consumer(output_context, &scan_output8, &scan_output16, NULL, &rgba64);
            scan_input += 4;
        }
    }

    return SAIL_OK;
}

static sail_status_t conversion_impl(const struct sail_image* image,
                                     struct sail_image* image_output,
                                     pixel_consumer_t pixel_consumer,
                                     int r, /* Index of the RED component.   */
                                     int g, /* Index of the GREEN component. */
                                     int b, /* Index of the BLUE component.  */
                                     int a, /* Index of the ALPHA component. */
                                     const struct sail_conversion_options* options)
{
    const struct output_context output_context = {image_output, r, g, b, a, options};

    /* After adding a new input pixel format, also update the switch in sail_can_convert(). */
    switch (image->pixel_format)
    {
    case SAIL_PIXEL_FORMAT_BPP1_INDEXED:
    {
        SAIL_TRY(convert_from_bpp1_indexed(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP2_INDEXED:
    {
        SAIL_TRY(convert_from_bpp2_indexed(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP4_INDEXED:
    {
        SAIL_TRY(convert_from_bpp4_indexed(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP8_INDEXED:
    {
        SAIL_TRY(convert_from_bpp8_indexed(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP1_GRAYSCALE:
    {
        SAIL_TRY(convert_from_bpp1_grayscale(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP2_GRAYSCALE:
    {
        SAIL_TRY(convert_from_bpp2_grayscale(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP4_GRAYSCALE:
    {
        SAIL_TRY(convert_from_bpp4_grayscale(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE:
    {
        SAIL_TRY(convert_from_bpp8_grayscale(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE:
    {
        SAIL_TRY(convert_from_bpp16_grayscale(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA:
    {
        SAIL_TRY(convert_from_bpp16_grayscale_alpha(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA:
    {
        SAIL_TRY(convert_from_bpp32_grayscale_alpha(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_RGB555:
    {
        SAIL_TRY(convert_from_bpp16_rgb555(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_BGR555:
    {
        SAIL_TRY(convert_from_bpp16_bgr555(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_RGB565:
    {
        SAIL_TRY(convert_from_bpp16_rgb565(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_BGR565:
    {
        SAIL_TRY(convert_from_bpp16_bgr565(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP24_RGB:
    {
        SAIL_TRY(convert_from_bpp24_rgb_kind(image, 0, 1, 2, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP24_BGR:
    {
        SAIL_TRY(convert_from_bpp24_rgb_kind(image, 2, 1, 0, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP48_RGB:
    {
        SAIL_TRY(convert_from_bpp48_rgb_kind(image, 0, 1, 2, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP48_BGR:
    {
        SAIL_TRY(convert_from_bpp48_rgb_kind(image, 2, 1, 0, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_RGBX:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 0, 1, 2, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_BGRX:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 2, 1, 0, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_XRGB:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 1, 2, 3, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_XBGR:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 3, 2, 1, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_RGBA:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 0, 1, 2, 3, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_BGRA:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 2, 1, 0, 3, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_ARGB:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 1, 2, 3, 0, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_ABGR:
    {
        SAIL_TRY(convert_from_bpp32_rgba_kind(image, 3, 2, 1, 0, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_RGBX:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 0, 1, 2, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_BGRX:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 2, 1, 0, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_XRGB:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 1, 2, 3, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_XBGR:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 3, 2, 1, -1, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_RGBA:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 0, 1, 2, 3, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_BGRA:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 2, 1, 0, 3, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_ARGB:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 1, 2, 3, 0, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_ABGR:
    {
        SAIL_TRY(convert_from_bpp64_rgba_kind(image, 3, 2, 1, 0, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_CMYK:
    {
        SAIL_TRY(convert_from_bpp32_cmyk(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_CMYK:
    {
        SAIL_TRY(convert_from_bpp64_cmyk(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP40_CMYKA:
    {
        SAIL_TRY(convert_from_bpp40_cmyka(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP80_CMYKA:
    {
        SAIL_TRY(convert_from_bpp80_cmyka(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP24_YCBCR:
    {
        SAIL_TRY(convert_from_bpp24_ycbcr(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_YCCK:
    {
        SAIL_TRY(convert_from_bpp32_ycck(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_HALF:
    {
        SAIL_TRY(convert_from_bpp16_grayscale_half(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_FLOAT:
    {
        SAIL_TRY(convert_from_bpp32_grayscale_float(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP48_RGB_HALF:
    {
        SAIL_TRY(convert_from_bpp48_rgb_half(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP64_RGBA_HALF:
    {
        SAIL_TRY(convert_from_bpp64_rgba_half(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT:
    {
        SAIL_TRY(convert_from_bpp96_rgb_float(image, pixel_consumer, &output_context));
        break;
    }
    case SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT:
    {
        SAIL_TRY(convert_from_bpp128_rgba_float(image, pixel_consumer, &output_context));
        break;
    }
    default:
    {
        SAIL_LOG_ERROR("Conversion from %s is not currently supported",
                       sail_pixel_format_to_string(image->pixel_format));
        SAIL_LOG_AND_RETURN(SAIL_ERROR_UNSUPPORTED_PIXEL_FORMAT);
    }
    }

    return SAIL_OK;
}

/*
 * Public functions.
 */

sail_status_t sail_convert_image(const struct sail_image* image,
                                 enum SailPixelFormat output_pixel_format,
                                 struct sail_image** image_output)
{
    SAIL_TRY(sail_convert_image_with_options(image, output_pixel_format, NULL /* options */, image_output));

    return SAIL_OK;
}

sail_status_t sail_convert_image_with_options(const struct sail_image* image,
                                              enum SailPixelFormat output_pixel_format,
                                              const struct sail_conversion_options* options,
                                              struct sail_image** image_output)
{
    SAIL_TRY(sail_check_image_valid(image));
    SAIL_CHECK_PTR(image_output);

    /* Handle conversion to indexed formats using quantization. */
    if (sail_is_indexed(output_pixel_format))
    {
        /* For indexed formats, first convert to RGB if needed. */
        struct sail_image* rgb_image = NULL;

        if (image->pixel_format != SAIL_PIXEL_FORMAT_BPP24_RGB && image->pixel_format != SAIL_PIXEL_FORMAT_BPP24_BGR
            && image->pixel_format != SAIL_PIXEL_FORMAT_BPP32_RGBA
            && image->pixel_format != SAIL_PIXEL_FORMAT_BPP32_BGRA
            && image->pixel_format != SAIL_PIXEL_FORMAT_BPP32_RGBX
            && image->pixel_format != SAIL_PIXEL_FORMAT_BPP32_BGRX)
        {
            SAIL_TRY(sail_convert_image_with_options(image, SAIL_PIXEL_FORMAT_BPP24_RGB, options, &rgb_image));
        }
        else
        {
            SAIL_TRY(sail_copy_image(image, &rgb_image));
        }

        /* Apply quantization with optional dithering. */
        bool dither = (options != NULL && (options->options & SAIL_CONVERSION_OPTION_DITHERING));
        SAIL_TRY_OR_CLEANUP(sail_quantize_image(rgb_image, output_pixel_format, dither, image_output),
                            /* cleanup */ sail_destroy_image(rgb_image));
        sail_destroy_image(rgb_image);

        return SAIL_OK;
    }

    int r, g, b, a;
    pixel_consumer_t pixel_consumer;
    SAIL_TRY(verify_and_construct_rgba_indexes_verbose(output_pixel_format, &pixel_consumer, &r, &g, &b, &a));

    struct sail_image* image_local;
    SAIL_TRY(sail_copy_image_skeleton(image, &image_local));

    image_local->pixel_format   = output_pixel_format;
    image_local->bytes_per_line = sail_bytes_per_line(image_local->width, image_local->pixel_format);

    /* Clear ICC profile if changing color space (RGB <-> Grayscale, CMYK <-> RGB, etc). */
    bool input_is_rgb    = sail_is_rgb_family(image->pixel_format);
    bool output_is_rgb   = sail_is_rgb_family(output_pixel_format);
    bool input_is_gray   = sail_is_grayscale(image->pixel_format);
    bool output_is_gray  = sail_is_grayscale(output_pixel_format);
    bool input_is_cmyk   = sail_is_cmyk(image->pixel_format);
    bool output_is_cmyk  = sail_is_cmyk(output_pixel_format);
    bool input_is_ycbcr  = sail_is_ycbcr(image->pixel_format);
    bool output_is_ycbcr = sail_is_ycbcr(output_pixel_format);
    bool input_is_ycck   = sail_is_ycck(image->pixel_format);
    bool output_is_ycck  = sail_is_ycck(output_pixel_format);

    if (input_is_rgb != output_is_rgb || input_is_gray != output_is_gray || input_is_cmyk != output_is_cmyk
        || input_is_ycbcr != output_is_ycbcr || input_is_ycck != output_is_ycck)
    {
        SAIL_LOG_DEBUG("Color space conversion detected, clearing ICC profile");
        sail_destroy_iccp(image_local->iccp);
        image_local->iccp = NULL;
    }

    const size_t pixels_size = (size_t)image_local->height * image_local->bytes_per_line;
    SAIL_TRY_OR_CLEANUP(sail_malloc(pixels_size, &image_local->pixels),
                        /* cleanup */ sail_destroy_image(image_local));

    /* Try fast-path conversion first (no alpha blending support in fast-path) */
    if (options == NULL || !(options->options & SAIL_CONVERSION_OPTION_BLEND_ALPHA))
    {
        if (sail_try_fast_conversion(image, image_local, output_pixel_format))
        {
            *image_output = image_local;
            return SAIL_OK;
        }
    }

    /* Fall back to standard conversion through intermediate RGBA */
    SAIL_TRY_OR_CLEANUP(conversion_impl(image, image_local, pixel_consumer, r, g, b, a, options),
                        /* cleanup */ sail_destroy_image(image_local));

    *image_output = image_local;

    return SAIL_OK;
}

sail_status_t sail_update_image(struct sail_image* image, enum SailPixelFormat output_pixel_format)
{
    SAIL_TRY(sail_update_image_with_options(image, output_pixel_format, NULL /* options */));

    return SAIL_OK;
}

sail_status_t sail_update_image_with_options(struct sail_image* image,
                                             enum SailPixelFormat output_pixel_format,
                                             const struct sail_conversion_options* options)
{
    SAIL_TRY(sail_check_image_valid(image));

    int r, g, b, a;
    pixel_consumer_t pixel_consumer;
    SAIL_TRY(verify_and_construct_rgba_indexes_verbose(output_pixel_format, &pixel_consumer, &r, &g, &b, &a));

    if (image->pixel_format == output_pixel_format)
    {
        return SAIL_OK;
    }

    const bool new_image_fits_into_existing =
        sail_greater_equal_bits_per_pixel(image->pixel_format, output_pixel_format);

    if (!new_image_fits_into_existing)
    {
        SAIL_LOG_ERROR("Updating from %s to %s cannot be done as the output is larger than the input",
                       sail_pixel_format_to_string(image->pixel_format),
                       sail_pixel_format_to_string(output_pixel_format));
        SAIL_LOG_AND_RETURN(SAIL_ERROR_UNSUPPORTED_PIXEL_FORMAT);
    }

    SAIL_TRY(conversion_impl(image, image, pixel_consumer, r, g, b, a, options));

    image->pixel_format = output_pixel_format;

    return SAIL_OK;
}

bool sail_can_convert(enum SailPixelFormat input_pixel_format, enum SailPixelFormat output_pixel_format)
{
    /* Special handling for indexed output formats (using quantization). */
    if (sail_is_indexed(output_pixel_format))
    {
        /* Any format that can convert to RGB can convert to indexed. */
        return sail_can_convert(input_pixel_format, SAIL_PIXEL_FORMAT_BPP24_RGB);
    }

    /* After adding a new input pixel format, also update the switch in conversion_impl(). */
    switch (input_pixel_format)
    {
    case SAIL_PIXEL_FORMAT_BPP1_INDEXED:
    case SAIL_PIXEL_FORMAT_BPP1_GRAYSCALE:
    case SAIL_PIXEL_FORMAT_BPP2_INDEXED:
    case SAIL_PIXEL_FORMAT_BPP2_GRAYSCALE:
    case SAIL_PIXEL_FORMAT_BPP4_INDEXED:
    case SAIL_PIXEL_FORMAT_BPP4_GRAYSCALE:
    case SAIL_PIXEL_FORMAT_BPP8_INDEXED:
    case SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE:
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE:
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA:
    case SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA:
    case SAIL_PIXEL_FORMAT_BPP16_RGB555:
    case SAIL_PIXEL_FORMAT_BPP16_BGR555:
    case SAIL_PIXEL_FORMAT_BPP16_RGB565:
    case SAIL_PIXEL_FORMAT_BPP16_BGR565:
    case SAIL_PIXEL_FORMAT_BPP24_RGB:
    case SAIL_PIXEL_FORMAT_BPP24_BGR:
    case SAIL_PIXEL_FORMAT_BPP48_RGB:
    case SAIL_PIXEL_FORMAT_BPP48_BGR:
    case SAIL_PIXEL_FORMAT_BPP32_RGBX:
    case SAIL_PIXEL_FORMAT_BPP32_BGRX:
    case SAIL_PIXEL_FORMAT_BPP32_XRGB:
    case SAIL_PIXEL_FORMAT_BPP32_XBGR:
    case SAIL_PIXEL_FORMAT_BPP32_RGBA:
    case SAIL_PIXEL_FORMAT_BPP32_BGRA:
    case SAIL_PIXEL_FORMAT_BPP32_ARGB:
    case SAIL_PIXEL_FORMAT_BPP32_ABGR:
    case SAIL_PIXEL_FORMAT_BPP64_RGBX:
    case SAIL_PIXEL_FORMAT_BPP64_BGRX:
    case SAIL_PIXEL_FORMAT_BPP64_XRGB:
    case SAIL_PIXEL_FORMAT_BPP64_XBGR:
    case SAIL_PIXEL_FORMAT_BPP64_RGBA:
    case SAIL_PIXEL_FORMAT_BPP64_BGRA:
    case SAIL_PIXEL_FORMAT_BPP64_ARGB:
    case SAIL_PIXEL_FORMAT_BPP64_ABGR:
    case SAIL_PIXEL_FORMAT_BPP32_CMYK:
    case SAIL_PIXEL_FORMAT_BPP64_CMYK:
    case SAIL_PIXEL_FORMAT_BPP40_CMYKA:
    case SAIL_PIXEL_FORMAT_BPP80_CMYKA:
    case SAIL_PIXEL_FORMAT_BPP24_YCBCR:
    case SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_HALF:
    case SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_FLOAT:
    case SAIL_PIXEL_FORMAT_BPP48_RGB_HALF:
    case SAIL_PIXEL_FORMAT_BPP64_RGBA_HALF:
    case SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT:
    case SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT:
    {
        int r, g, b, a;
        pixel_consumer_t pixel_consumer;
        return verify_and_construct_rgba_indexes_silent(output_pixel_format, &pixel_consumer, &r, &g, &b, &a);
    }
    default:
    {
        return false;
    }
    }
}

/* Sorted by priority. */
static const enum SailPixelFormat GRAYSCALE_CANDIDATES[] = {
    /* After adding a new output pixel format, also update this list. */
    SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE,
    SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE,

    SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE_ALPHA,
    SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA,
    SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA,

    SAIL_PIXEL_FORMAT_BPP24_YCBCR,
    SAIL_PIXEL_FORMAT_BPP24_YUV,

    SAIL_PIXEL_FORMAT_BPP24_RGB,
    SAIL_PIXEL_FORMAT_BPP24_BGR,

    SAIL_PIXEL_FORMAT_BPP48_RGB,
    SAIL_PIXEL_FORMAT_BPP48_BGR,

    SAIL_PIXEL_FORMAT_BPP16_RGB555,
    SAIL_PIXEL_FORMAT_BPP16_BGR555,
    SAIL_PIXEL_FORMAT_BPP16_RGB565,
    SAIL_PIXEL_FORMAT_BPP16_BGR565,

    SAIL_PIXEL_FORMAT_BPP32_RGBA,
    SAIL_PIXEL_FORMAT_BPP32_BGRA,
    SAIL_PIXEL_FORMAT_BPP32_ARGB,
    SAIL_PIXEL_FORMAT_BPP32_ABGR,
    SAIL_PIXEL_FORMAT_BPP32_RGBX,
    SAIL_PIXEL_FORMAT_BPP32_BGRX,
    SAIL_PIXEL_FORMAT_BPP32_XRGB,
    SAIL_PIXEL_FORMAT_BPP32_XBGR,

    SAIL_PIXEL_FORMAT_BPP64_RGBA,
    SAIL_PIXEL_FORMAT_BPP64_BGRA,
    SAIL_PIXEL_FORMAT_BPP64_ARGB,
    SAIL_PIXEL_FORMAT_BPP64_ABGR,
    SAIL_PIXEL_FORMAT_BPP64_RGBX,
    SAIL_PIXEL_FORMAT_BPP64_BGRX,
    SAIL_PIXEL_FORMAT_BPP64_XRGB,
    SAIL_PIXEL_FORMAT_BPP64_XBGR,

    SAIL_PIXEL_FORMAT_BPP16_RGBA,
    SAIL_PIXEL_FORMAT_BPP16_BGRA,
    SAIL_PIXEL_FORMAT_BPP16_ARGB,
    SAIL_PIXEL_FORMAT_BPP16_ABGR,
    SAIL_PIXEL_FORMAT_BPP16_RGBX,
    SAIL_PIXEL_FORMAT_BPP16_BGRX,
    SAIL_PIXEL_FORMAT_BPP16_XRGB,
    SAIL_PIXEL_FORMAT_BPP16_XBGR,
};

static const size_t GRAYSCALE_CANDIDATES_LENGTH = sizeof(GRAYSCALE_CANDIDATES) / sizeof(GRAYSCALE_CANDIDATES[0]);

/* Sorted by priority. */
static const enum SailPixelFormat INDEXED_OR_FULL_COLOR_CANDIDATES[] = {
    /* After adding a new output pixel format, also update this list. */
    SAIL_PIXEL_FORMAT_BPP24_YCBCR,
    SAIL_PIXEL_FORMAT_BPP24_YUV,

    SAIL_PIXEL_FORMAT_BPP24_RGB,
    SAIL_PIXEL_FORMAT_BPP24_BGR,

    SAIL_PIXEL_FORMAT_BPP48_RGB,
    SAIL_PIXEL_FORMAT_BPP48_BGR,

    SAIL_PIXEL_FORMAT_BPP16_RGB555,
    SAIL_PIXEL_FORMAT_BPP16_BGR555,
    SAIL_PIXEL_FORMAT_BPP16_RGB565,
    SAIL_PIXEL_FORMAT_BPP16_BGR565,

    SAIL_PIXEL_FORMAT_BPP32_RGBA,
    SAIL_PIXEL_FORMAT_BPP32_BGRA,
    SAIL_PIXEL_FORMAT_BPP32_ARGB,
    SAIL_PIXEL_FORMAT_BPP32_ABGR,
    SAIL_PIXEL_FORMAT_BPP32_RGBX,
    SAIL_PIXEL_FORMAT_BPP32_BGRX,
    SAIL_PIXEL_FORMAT_BPP32_XRGB,
    SAIL_PIXEL_FORMAT_BPP32_XBGR,

    SAIL_PIXEL_FORMAT_BPP64_RGBA,
    SAIL_PIXEL_FORMAT_BPP64_BGRA,
    SAIL_PIXEL_FORMAT_BPP64_ARGB,
    SAIL_PIXEL_FORMAT_BPP64_ABGR,
    SAIL_PIXEL_FORMAT_BPP64_RGBX,
    SAIL_PIXEL_FORMAT_BPP64_BGRX,
    SAIL_PIXEL_FORMAT_BPP64_XRGB,
    SAIL_PIXEL_FORMAT_BPP64_XBGR,

    SAIL_PIXEL_FORMAT_BPP16_RGBA,
    SAIL_PIXEL_FORMAT_BPP16_BGRA,
    SAIL_PIXEL_FORMAT_BPP16_ARGB,
    SAIL_PIXEL_FORMAT_BPP16_ABGR,
    SAIL_PIXEL_FORMAT_BPP16_RGBX,
    SAIL_PIXEL_FORMAT_BPP16_BGRX,
    SAIL_PIXEL_FORMAT_BPP16_XRGB,
    SAIL_PIXEL_FORMAT_BPP16_XBGR,

    SAIL_PIXEL_FORMAT_BPP32_CMYK,
    SAIL_PIXEL_FORMAT_BPP64_CMYK,

    SAIL_PIXEL_FORMAT_BPP8_INDEXED,
    SAIL_PIXEL_FORMAT_BPP4_INDEXED,
    SAIL_PIXEL_FORMAT_BPP2_INDEXED,
    SAIL_PIXEL_FORMAT_BPP1_INDEXED,

    SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE,
    SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE,

    SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE_ALPHA,
    SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA,
    SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA,
};

static const size_t INDEXED_OR_FULL_COLOR_CANDIDATES_LENGTH =
    sizeof(INDEXED_OR_FULL_COLOR_CANDIDATES) / sizeof(INDEXED_OR_FULL_COLOR_CANDIDATES[0]);

enum SailPixelFormat sail_closest_pixel_format(enum SailPixelFormat input_pixel_format,
                                               const enum SailPixelFormat pixel_formats[],
                                               size_t pixel_formats_length)
{
    if (input_pixel_format == SAIL_PIXEL_FORMAT_UNKNOWN)
    {
        return SAIL_PIXEL_FORMAT_UNKNOWN;
    }

    const enum SailPixelFormat* candidates;
    size_t candidates_length;

    if (sail_is_grayscale(input_pixel_format))
    {
        candidates        = GRAYSCALE_CANDIDATES;
        candidates_length = GRAYSCALE_CANDIDATES_LENGTH;
    }
    else
    {
        candidates        = INDEXED_OR_FULL_COLOR_CANDIDATES;
        candidates_length = INDEXED_OR_FULL_COLOR_CANDIDATES_LENGTH;
    }

    size_t best_index_candidate = UINT_MAX;
    size_t best_index_result    = 0;
    bool found                  = false;

    /* O(n^2) (sic!). */
    for (size_t i = 0; i < pixel_formats_length; i++)
    {
        for (size_t k = 0; k < candidates_length; k++)
        {
            if (pixel_formats[i] == candidates[k])
            {
                if (k < best_index_candidate)
                {
                    best_index_candidate = k;
                    best_index_result    = i;
                    found                = true;
                    break;
                }
            }
        }
    }

    return found ? pixel_formats[best_index_result] : SAIL_PIXEL_FORMAT_UNKNOWN;
}

enum SailPixelFormat sail_closest_pixel_format_from_save_features(enum SailPixelFormat input_pixel_format,
                                                                  const struct sail_save_features* save_features)
{
    return sail_closest_pixel_format(input_pixel_format, save_features->pixel_formats,
                                     save_features->pixel_formats_length);
}

sail_status_t sail_convert_image_for_saving(const struct sail_image* image,
                                            const struct sail_save_features* save_features,
                                            struct sail_image** image_output)
{
    SAIL_TRY(sail_convert_image_for_saving_with_options(image, save_features, NULL, image_output));

    return SAIL_OK;
}

sail_status_t sail_convert_image_for_saving_with_options(const struct sail_image* image,
                                                         const struct sail_save_features* save_features,
                                                         const struct sail_conversion_options* options,
                                                         struct sail_image** image_output)
{
    SAIL_TRY(sail_check_image_valid(image));
    SAIL_CHECK_PTR(save_features);
    SAIL_CHECK_PTR(image_output);

    enum SailPixelFormat best_pixel_format =
        sail_closest_pixel_format_from_save_features(image->pixel_format, save_features);

    if (best_pixel_format == SAIL_PIXEL_FORMAT_UNKNOWN)
    {
        SAIL_LOG_ERROR("Failed to find the best output format for saving %s image",
                       sail_pixel_format_to_string(image->pixel_format));
        SAIL_LOG_AND_RETURN(SAIL_ERROR_UNSUPPORTED_PIXEL_FORMAT);
    }

    if (best_pixel_format == image->pixel_format)
    {
        SAIL_TRY(sail_copy_image(image, image_output));
    }
    else
    {
        SAIL_TRY(sail_convert_image_with_options(image, best_pixel_format, options, image_output));
    }

    return SAIL_OK;
}
