/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2023      Yi-Yen Chung <eric681@andestech.com> (Copyright owned by Andes Technology)
 */

#define SIMDE_TEST_ARM_NEON_INSN abal_high

#include "test-neon.h"
#include <simde/arm/neon/abal_high.h>

/* https://community.intel.com/t5/Intel-C-Compiler/ICC-generates-incorrect-code/td-p/1199261 */
#if defined(HEDLEY_INTEL_VERSION)
#  define TEST_SIMDE_VABD_NO_TEST_32
#endif

static int
test_simde_vabal_high_s8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    int16_t a[8];
    int8_t b[16];
    int8_t c[16];
    int16_t r[8];
  } test_vec[] = {
    { { INT16_C(716), INT16_C(29151), INT16_C(29093), INT16_C(27271), INT16_C(5789), INT16_C(17316), -INT16_C(8236), INT16_C(18621) },
      { -INT8_C(127), INT8_C(5), -INT8_C(115), -INT8_C(102), INT8_C(43), -INT8_C(53), INT8_C(15), -INT8_C(4),
        -INT8_C(9), INT8_C(37), -INT8_C(117), INT8_C(76), -INT8_C(22), INT8_C(32), INT8_C(94), INT8_C(78) },
      { -INT8_C(11), INT8_C(96), INT8_C(65), INT8_C(60), -INT8_C(96), INT8_C(35), INT8_C(83), -INT8_C(38),
        INT8_C(20), -INT8_C(82), -INT8_C(105), -INT8_C(122), INT8_C(52), INT8_C(108), INT8_C(56), INT8_C(97) },
      { INT16_C(745), INT16_C(29270), INT16_C(29105), INT16_C(27469), INT16_C(5863), INT16_C(17392), -INT16_C(8198), INT16_C(18640) } },
    { { -INT16_C(24748), -INT16_C(20075), -INT16_C(16319), INT16_C(28770), -INT16_C(3900), -INT16_C(31101), -INT16_C(6390), INT16_C(14598) },
      { -INT8_C(91), INT8_C(30), -INT8_C(76), -INT8_C(44), INT8_C(43), INT8_C(113), INT8_C(62), INT8_C(52),
        INT8_C(51), -INT8_C(97), INT8_C(15), INT8_C(116), -INT8_C(77), INT8_C(76), INT8_C(2), INT8_C(74) },
      { INT8_C(82), INT8_C(83), INT8_C(16), INT8_C(6), -INT8_C(77), -INT8_C(29), -INT8_C(83), INT8_C(53),
        -INT8_C(14), -INT8_C(75), -INT8_C(49), INT8_C(90), INT8_C(13), INT8_C(112), INT8_C(58), -INT8_C(95) },
      { -INT16_C(24683), -INT16_C(20053), -INT16_C(16255), INT16_C(28796), -INT16_C(3810), -INT16_C(31065), -INT16_C(6334), INT16_C(14767) } },
    { { -INT16_C(13373), -INT16_C(22548), -INT16_C(22456), -INT16_C(25197), -INT16_C(27484), INT16_C(8486), INT16_C(7004), -INT16_C(832) },
      { INT8_C(59), INT8_C(13), -INT8_C(123), INT8_C(127), -INT8_C(111), INT8_C(122), INT8_C(45), -INT8_C(26),
        INT8_C(65), -INT8_C(40), -INT8_C(112), -INT8_C(97), -INT8_C(86), -INT8_C(53), -INT8_C(96), -INT8_C(85) },
      { -INT8_C(72), INT8_C(74), INT8_C(64), INT8_C(127), -INT8_C(109), -INT8_C(75), INT8_C(112), INT8_C(124),
        -INT8_C(77), INT8_C(88), -INT8_C(106), INT8_C(5), -INT8_C(23), INT8_C(10), -INT8_C(112), -INT8_C(6) },
      { -INT16_C(13231), -INT16_C(22420), -INT16_C(22450), -INT16_C(25095), -INT16_C(27421), INT16_C(8549), INT16_C(7020), -INT16_C(753) } },
    { { INT16_C(30356), INT16_C(19481), INT16_C(17652), INT16_C(22560), -INT16_C(8359), -INT16_C(5853), INT16_C(17494), INT16_C(26669) },
      { INT8_C(55), -INT8_C(95), -INT8_C(109), -INT8_C(96), INT8_C(50), INT8_C(66), INT8_C(106), INT8_C(37),
        INT8_C(66), INT8_C(116), -INT8_C(40), -INT8_C(51), -INT8_C(89), -INT8_C(34), -INT8_C(92), -INT8_C(11) },
      { INT8_C(113), INT8_C(77), -INT8_C(118), -INT8_C(82), -INT8_C(71), -INT8_C(92), INT8_C(110), INT8_C(53),
        -INT8_C(64), INT8_C(127), -INT8_C(102), -INT8_C(53), -INT8_C(27), -INT8_C(47), -INT8_C(20), INT8_C(108) },
      { INT16_C(30486), INT16_C(19492), INT16_C(17714), INT16_C(22562), -INT16_C(8297), -INT16_C(5840), INT16_C(17566), INT16_C(26788) } },
    { { INT16_C(19788), -INT16_C(17957), -INT16_C(22695), INT16_C(27620), -INT16_C(18937), -INT16_C(15606), -INT16_C(17612), -INT16_C(16042) },
      { -INT8_C(83), INT8_C(86), -INT8_C(41), INT8_C(84), -INT8_C(110), -INT8_C(19), INT8_C(54), INT8_C(80),
        INT8_C(15), INT8_C(125), INT8_C(118), -INT8_C(77), -INT8_C(45), INT8_C(60), INT8_C(54), -INT8_C(36) },
      { -INT8_C(110), INT8_C(40), -INT8_C(108), INT8_C(16), INT8_C(111), INT8_C(41), -INT8_C(44), -INT8_C(96),
        INT8_C(46), INT8_C(15), -INT8_C(57), -INT8_C(40), -INT8_C(105), -INT8_C(82), INT8_C(22), INT8_C(104) },
      { INT16_C(19819), -INT16_C(17847), -INT16_C(22520), INT16_C(27657), -INT16_C(18877), -INT16_C(15464), -INT16_C(17580), -INT16_C(15902) } },
    { { -INT16_C(13139), -INT16_C(15416), -INT16_C(9595), -INT16_C(25280), INT16_C(28260), INT16_C(24587), INT16_C(22167), -INT16_C(19553) },
      { -INT8_C(78), INT8_C(44), INT8_C(116), INT8_C(81), -INT8_C(30), -INT8_C(102), -INT8_C(28), INT8_C(28),
        -INT8_C(10), -INT8_C(126), -INT8_C(110), -INT8_C(64), INT8_C(20), -INT8_C(46), -INT8_C(49), INT8_C(62) },
      { -INT8_C(112), INT8_C(115), INT8_C(12), -INT8_C(49), -INT8_C(92), INT8_C(47), -INT8_C(45), -INT8_C(125),
        -INT8_C(40), INT8_C(28), INT8_C(49), -INT8_C(35), -INT8_C(76), -INT8_C(107), INT8_C(66), INT8_C(92) },
      { -INT16_C(13109), -INT16_C(15262), -INT16_C(9436), -INT16_C(25251), INT16_C(28356), INT16_C(24648), INT16_C(22282), -INT16_C(19523) } },
    { { -INT16_C(19282), INT16_C(22460), -INT16_C(27815), INT16_C(29281), INT16_C(22043), -INT16_C(12003), -INT16_C(13572), INT16_C(30020) },
      { INT8_C(70), -INT8_C(4), -INT8_C(42), -INT8_C(37), INT8_C(94), INT8_C(121), INT8_C(25), -INT8_C(72),
        INT8_C(61), INT8_C(10), INT8_C(83), INT8_C(52), INT8_C(1), INT8_C(34), INT8_C(35), -INT8_C(66) },
      { -INT8_C(67), -INT8_C(78), -INT8_C(34), INT8_C(7), -INT8_C(128), -INT8_C(19), INT8_C(4), INT8_C(78),
        -INT8_C(99), -INT8_C(115), INT8_C(94), -INT8_C(64), -INT8_C(12), INT8_C(21), -INT8_C(25), INT8_C(66) },
      { -INT16_C(19122), INT16_C(22585), -INT16_C(27804), INT16_C(29397), INT16_C(22056), -INT16_C(11990), -INT16_C(13512), INT16_C(30152) } },
    { { -INT16_C(7538), INT16_C(25602), INT16_C(24299), -INT16_C(11918), INT16_C(3229), -INT16_C(18055), INT16_C(32599), -INT16_C(15279) },
      { -INT8_C(107), INT8_C(66), -INT8_C(115), -INT8_C(100), INT8_C(98), -INT8_C(32), -INT8_C(13), INT8_C(39),
        INT8_C(55), -INT8_C(47), -INT8_C(35), INT8_C(28), INT8_C(38), INT8_C(110), INT8_C(44), -INT8_C(29) },
      { INT8_C(116), -INT8_C(101), INT8_C(109), INT8_C(75), INT8_C(39), INT8_C(23), INT8_C(19), -INT8_C(77),
        INT8_C(95), -INT8_C(18), -INT8_C(74), INT8_C(35), -INT8_C(113), -INT8_C(79), INT8_C(29), -INT8_C(29) },
      { -INT16_C(7498), INT16_C(25631), INT16_C(24338), -INT16_C(11911), INT16_C(3380), -INT16_C(17866), INT16_C(32614), -INT16_C(15279) } },
    { { INT16_C(27758), INT16_C(22488), INT16_C(6395), INT16_C(25945), INT16_C(28102), -INT16_C(14137), INT16_C(28301), INT16_C(24371) },
      { -INT8_C(63), INT8_C(54), -INT8_C(112), INT8_C(55), INT8_C(99), -INT8_C(70), -INT8_C(84), -INT8_C(116),
        -INT8_C(117), -INT8_C(30), INT8_C(42), -INT8_C(45), -INT8_C(12), -INT8_C(3), -INT8_C(80), INT8_C(54) },
      { INT8_C(112), -INT8_C(126), -INT8_C(94), -INT8_C(123), INT8_C(122), INT8_C(16), -INT8_C(23), INT8_C(28),
        INT8_C(74), -INT8_C(79), INT8_C(125), -INT8_C(71), -INT8_C(90), -INT8_C(93), -INT8_C(63), INT8_C(115) },
      { INT16_C(27949), INT16_C(22537), INT16_C(6478), INT16_C(25971), INT16_C(28180), -INT16_C(14047), INT16_C(28318), INT16_C(24432) } },
    { { -INT16_C(3775), INT16_C(28998), INT16_C(27693), -INT16_C(15256), INT16_C(28027), -INT16_C(32728), -INT16_C(23584), -INT16_C(9778) },
      { INT8_C(112), INT8_C(80), INT8_C(101), -INT8_C(10), INT8_C(21), -INT8_C(23), INT8_C(6), -INT8_C(101),
        -INT8_C(83), -INT8_C(50), INT8_C(87), -INT8_C(58), -INT8_C(37), INT8_C(15), INT8_C(55), -INT8_C(63) },
      { -INT8_C(7), -INT8_C(3), INT8_C(31), INT8_C(94), -INT8_C(103), INT8_C(23), INT8_C(17), INT8_C(62),
        -INT8_C(15), INT8_C(46), INT8_C(4), -INT8_C(104), INT8_C(30), INT8_C(51), -INT8_C(38), INT8_C(9) },
      { -INT16_C(3707), INT16_C(29094), INT16_C(27776), -INT16_C(15210), INT16_C(28094), -INT16_C(32692), -INT16_C(23491), -INT16_C(9706) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int16x8_t a = simde_vld1q_s16(test_vec[i].a);
    simde_int8x16_t b = simde_vld1q_s8(test_vec[i].b);
    simde_int8x16_t c = simde_vld1q_s8(test_vec[i].c);
    simde_int16x8_t r = simde_vabal_high_s8(a, b, c);

    simde_test_arm_neon_assert_equal_i16x8(r, simde_vld1q_s16(test_vec[i].r));
  }

  return 0;
}
static int
test_simde_vabal_high_s16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    int32_t a[4];
    int16_t b[8];
    int16_t c[8];
    int32_t r[4];
  } test_vec[] = {
    { { -INT32_C(179829637), INT32_C(188529379), INT32_C(288350722), -INT32_C(517571286) },
      { -INT16_C(13612), -INT16_C(28223), -INT16_C(7027), -INT16_C(15233), -INT16_C(10290), -INT16_C(24042), INT16_C(23927), -INT16_C(25125) },
      { -INT16_C(784), INT16_C(28117), INT16_C(21530), INT16_C(9983), -INT16_C(13365), INT16_C(4343), -INT16_C(30122), INT16_C(21887) },
      { -INT32_C(179826562), INT32_C(188557764), INT32_C(288404771), -INT32_C(517524274) } },
    { { INT32_C(196428168), INT32_C(174905394), -INT32_C(1357007296), INT32_C(1621511496) },
      { INT16_C(18263), -INT16_C(32488), INT16_C(395), INT16_C(1235), INT16_C(28962), INT16_C(31407), -INT16_C(15134), INT16_C(17014) },
      { INT16_C(17062), INT16_C(24153), -INT16_C(7606), INT16_C(17212), -INT16_C(7826), -INT16_C(6303), -INT16_C(11693), INT16_C(17372) },
      { INT32_C(196464956), INT32_C(174943104), -INT32_C(1357003855), INT32_C(1621511854) } },
    { { -INT32_C(804161572), -INT32_C(1193738346), INT32_C(429598036), INT32_C(136395532) },
      { INT16_C(31912), INT16_C(22293), INT16_C(25268), -INT16_C(32551), INT16_C(1997), -INT16_C(651), INT16_C(14691), -INT16_C(14518) },
      { INT16_C(1671), -INT16_C(9415), INT16_C(5287), -INT16_C(19075), INT16_C(14489), -INT16_C(16213), INT16_C(21249), -INT16_C(20147) },
      { -INT32_C(804149080), -INT32_C(1193722784), INT32_C(429604594), INT32_C(136401161) } },
    { { INT32_C(1486856531), INT32_C(1112849963), -INT32_C(4448778), INT32_C(1472436894) },
      { INT16_C(19724), INT16_C(28219), -INT16_C(9804), INT16_C(18085), INT16_C(31900), -INT16_C(16435), INT16_C(9764), -INT16_C(29100) },
      { -INT16_C(23372), -INT16_C(30253), -INT16_C(23622), -INT16_C(27362), -INT16_C(18517), -INT16_C(32112), -INT16_C(17103), -INT16_C(23338) },
      { INT32_C(1486906948), INT32_C(1112865640), -INT32_C(4421911), INT32_C(1472442656) } },
    { { INT32_C(529456900), -INT32_C(112049144), INT32_C(2082647990), INT32_C(104254184) },
      { INT16_C(26185), -INT16_C(9328), -INT16_C(19245), INT16_C(20334), -INT16_C(7030), INT16_C(9092), -INT16_C(16424), -INT16_C(12426) },
      { -INT16_C(25020), -INT16_C(7843), -INT16_C(21867), INT16_C(32706), INT16_C(9564), INT16_C(28046), -INT16_C(19223), -INT16_C(25715) },
      { INT32_C(529473494), -INT32_C(112030190), INT32_C(2082650789), INT32_C(104267473) } },
    { { -INT32_C(2038153826), INT32_C(348526704), -INT32_C(1695251148), INT32_C(309568932) },
      { -INT16_C(21078), -INT16_C(24096), -INT16_C(12069), INT16_C(7162), -INT16_C(5108), INT16_C(26854), INT16_C(32114), INT16_C(29269) },
      { INT16_C(30475), -INT16_C(12380), INT16_C(22264), INT16_C(12142), INT16_C(10088), -INT16_C(32109), INT16_C(17963), -INT16_C(15000) },
      { -INT32_C(2038138630), INT32_C(348585667), -INT32_C(1695236997), INT32_C(309613201) } },
    { { INT32_C(761698564), -INT32_C(426557488), INT32_C(1608948529), INT32_C(1963604586) },
      { -INT16_C(8301), -INT16_C(19858), -INT16_C(29881), INT16_C(21354), INT16_C(19475), -INT16_C(10728), INT16_C(2436), -INT16_C(31490) },
      { -INT16_C(14416), -INT16_C(20272), -INT16_C(12400), INT16_C(7982), -INT16_C(19487), INT16_C(11576), INT16_C(19964), -INT16_C(15614) },
      { INT32_C(761737526), -INT32_C(426535184), INT32_C(1608966057), INT32_C(1963620462) } },
    { { -INT32_C(1630703034), -INT32_C(1886177776), -INT32_C(852485065), -INT32_C(951651993) },
      { INT16_C(23651), INT16_C(7004), -INT16_C(656), -INT16_C(6092), -INT16_C(28478), -INT16_C(12117), -INT16_C(5892), -INT16_C(29227) },
      { INT16_C(32449), -INT16_C(21110), -INT16_C(1208), -INT16_C(16097), INT16_C(152), INT16_C(3396), INT16_C(20552), INT16_C(23176) },
      { -INT32_C(1630674404), -INT32_C(1886162263), -INT32_C(852458621), -INT32_C(951599590) } },
    { { -INT32_C(1596752989), INT32_C(2103359502), INT32_C(57581753), -INT32_C(222500531) },
      { INT16_C(111), INT16_C(2594), -INT16_C(22361), -INT16_C(15272), INT16_C(18492), -INT16_C(4773), INT16_C(31736), INT16_C(13386) },
      { -INT16_C(3931), -INT16_C(27736), -INT16_C(15212), INT16_C(19636), INT16_C(19625), -INT16_C(13211), -INT16_C(26545), INT16_C(30610) },
      { -INT32_C(1596751856), INT32_C(2103367940), INT32_C(57640034), -INT32_C(222483307) } },
    { { -INT32_C(607395563), -INT32_C(409695611), -INT32_C(744659695), INT32_C(458233591) },
      { -INT16_C(20088), -INT16_C(26599), -INT16_C(1512), INT16_C(18133), -INT16_C(31527), -INT16_C(22654), -INT16_C(16130), INT16_C(16565) },
      { -INT16_C(5175), INT16_C(16754), INT16_C(27895), INT16_C(30779), INT16_C(12703), -INT16_C(16165), INT16_C(31881), INT16_C(3404) },
      { -INT32_C(607351333), -INT32_C(409689122), -INT32_C(744611684), INT32_C(458246752) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int32x4_t a = simde_vld1q_s32(test_vec[i].a);
    simde_int16x8_t b = simde_vld1q_s16(test_vec[i].b);
    simde_int16x8_t c = simde_vld1q_s16(test_vec[i].c);
    simde_int32x4_t r = simde_vabal_high_s16(a, b, c);

    simde_test_arm_neon_assert_equal_i32x4(r, simde_vld1q_s32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_vabal_high_s32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    int64_t a[2];
    int32_t b[4];
    int32_t c[4];
    int64_t r[2];
  } test_vec[] = {
    { { -INT64_C(8335084918904227682), INT64_C(5082616870088972360) },
      { -INT32_C(1203938446), -INT32_C(1379772492), INT32_C(616994359), INT32_C(1083545082) },
      { -INT32_C(1906729359), INT32_C(1015093241), INT32_C(425590947), INT32_C(2029372009) },
      { -INT64_C(8335084918712824270), INT64_C(5082616871034799287) } },
    { { -INT64_C(6692201385341432215), INT64_C(6130272179971188291) },
      { -INT32_C(57658811), INT32_C(326008941), INT32_C(1741884817), -INT32_C(684168561) },
      { -INT32_C(970266010), -INT32_C(1878151217), INT32_C(728958119), INT32_C(18297972) },
      { -INT64_C(6692201384328505517), INT64_C(6130272180673654824) } },
    { { -INT64_C(3228547053428818110), -INT64_C(3775920480530457269) },
      { -INT32_C(1896729348), -INT32_C(1340240637), -INT32_C(252156445), INT32_C(1199317340) },
      { INT32_C(1266695928), INT32_C(1007266895), INT32_C(1686726418), INT32_C(2083350046) },
      { -INT64_C(3228547051489935247), -INT64_C(3775920479646424563) } },
    { { -INT64_C(5079326313705028170), INT64_C(509341014050276175) },
      { -INT32_C(2012068063), -INT32_C(1347573040), -INT32_C(750124298), INT32_C(1242375513) },
      { INT32_C(1262690307), INT32_C(1793471940), -INT32_C(1987031715), -INT32_C(1187341857) },
      { -INT64_C(5079326312468120753), INT64_C(509341016479993545) } },
    { { -INT64_C(910957854149526510), -INT64_C(1899348721453545205) },
      { -INT32_C(1550797276), -INT32_C(878055843), -INT32_C(1270476620), -INT32_C(1674582391) },
      { -INT32_C(720711827), -INT32_C(303966234), -INT32_C(1744041663), INT32_C(44808599) },
      { -INT64_C(910957853675961467), -INT64_C(1899348719734154215) } },
    { { INT64_C(1577253342371792335), -INT64_C(555804675587586129) },
      { -INT32_C(320249636), INT32_C(1372770730), -INT32_C(177109545), INT32_C(579434157) },
      { INT32_C(599026848), -INT32_C(36369713), INT32_C(685560212), -INT32_C(990132316) },
      { INT64_C(1577253343234462092), -INT64_C(555804674018019656) } },
    { { -INT64_C(1253293116004059154), INT64_C(3646315862760396425) },
      { INT32_C(1622631446), INT32_C(355513214), INT32_C(437168953), INT32_C(218678106) },
      { -INT32_C(209040040), INT32_C(534132610), -INT32_C(666122681), INT32_C(337309625) },
      { -INT64_C(1253293114900767520), INT64_C(3646315862879027944) } },
    { { INT64_C(2045382784511940269), INT64_C(6639743016461780648) },
      { -INT32_C(365809482), INT32_C(468234551), -INT32_C(1114166296), INT32_C(945962830) },
      { -INT32_C(2036305755), INT32_C(147964901), INT32_C(401209098), INT32_C(989423590) },
      { INT64_C(2045382786027315663), INT64_C(6639743016505241408) } },
    { { INT64_C(3621735817771913007), -INT64_C(19392873471066367) },
      { -INT32_C(563773313), INT32_C(1015072499), -INT32_C(938101529), INT32_C(184654273) },
      { -INT32_C(1212023151), -INT32_C(1740549979), INT32_C(585644788), INT32_C(2081175347) },
      { INT64_C(3621735819295659324), -INT64_C(19392871574545293) } },
    { { INT64_C(948366791356237313), INT64_C(8912436025019436909) },
      { INT32_C(420891144), INT32_C(1780120745), -INT32_C(219505428), INT32_C(1015328086) },
      { INT32_C(292113003), -INT32_C(2041758122), -INT32_C(1731207938), -INT32_C(1516599938) },
      { INT64_C(948366792867939823), INT64_C(8912436027551364933) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_int64x2_t a = simde_vld1q_s64(test_vec[i].a);
    simde_int32x4_t b = simde_vld1q_s32(test_vec[i].b);
    simde_int32x4_t c = simde_vld1q_s32(test_vec[i].c);
    simde_int64x2_t r = simde_vabal_high_s32(a, b, c);

    simde_test_arm_neon_assert_equal_i64x2(r, simde_vld1q_s64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_vabal_high_u8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    uint16_t a[8];
    uint8_t b[16];
    uint8_t c[16];
    uint16_t r[8];
  } test_vec[] = {
    { { UINT16_C(38828), UINT16_C(41435), UINT16_C(57053), UINT16_C(49725), UINT16_C(5202), UINT16_C(59645), UINT16_C(41047), UINT16_C(45717) },
      { UINT8_C(13), UINT8_C(113), UINT8_C(222), UINT8_C(170), UINT8_C(90), UINT8_C(239), UINT8_C(232), UINT8_C(191),
        UINT8_C(145), UINT8_C(156), UINT8_C(183), UINT8_C(214), UINT8_C(44), UINT8_C(59), UINT8_C(171), UINT8_C(161) },
      { UINT8_C(44), UINT8_C(117), UINT8_C(34), UINT8_C(97), UINT8_C(162), UINT8_C(73), UINT8_C(214), UINT8_C(138),
        UINT8_C(140), UINT8_C(240), UINT8_C(5), UINT8_C(84), UINT8_C(106), UINT8_C(130), UINT8_C(252), UINT8_C(155) },
      { UINT16_C(38833), UINT16_C(41519), UINT16_C(57231), UINT16_C(49855), UINT16_C(5264), UINT16_C(59716), UINT16_C(41128), UINT16_C(45723) } },
    { { UINT16_C(55952), UINT16_C(63200), UINT16_C(36745), UINT16_C(50405), UINT16_C(17176), UINT16_C(54849), UINT16_C(61966), UINT16_C(28342) },
      { UINT8_C(200), UINT8_C(221), UINT8_C(187), UINT8_C(41), UINT8_C(3), UINT8_C(14), UINT8_C(20), UINT8_C(8),
        UINT8_C(131), UINT8_C(18), UINT8_C(68), UINT8_C(156), UINT8_C(53), UINT8_C(243), UINT8_C(97), UINT8_C(145) },
      { UINT8_C(205), UINT8_C(31), UINT8_C(190), UINT8_C(113), UINT8_C(237), UINT8_C(252), UINT8_C(169), UINT8_C(111),
        UINT8_C(37), UINT8_C(167), UINT8_C(95), UINT8_C(94), UINT8_C(85), UINT8_C(6), UINT8_C(137), UINT8_C(183) },
      { UINT16_C(56046), UINT16_C(63349), UINT16_C(36772), UINT16_C(50467), UINT16_C(17208), UINT16_C(55086), UINT16_C(62006), UINT16_C(28380) } },
    { { UINT16_C(44990), UINT16_C(61661), UINT16_C(22479), UINT16_C(38536), UINT16_C(28216), UINT16_C(21945), UINT16_C(14021), UINT16_C(33738) },
      { UINT8_C(204), UINT8_C(55), UINT8_C(32), UINT8_C(29), UINT8_C(239), UINT8_C(201), UINT8_C(173), UINT8_C(204),
        UINT8_C(163), UINT8_C(193), UINT8_C(131), UINT8_C(174), UINT8_C(42), UINT8_C(209), UINT8_C(212), UINT8_C(166) },
      { UINT8_C(196), UINT8_C(175), UINT8_C(189), UINT8_C(6), UINT8_C(157), UINT8_C(244), UINT8_C(102), UINT8_C(191),
        UINT8_C(82), UINT8_C(123), UINT8_C(60), UINT8_C(158), UINT8_C(166), UINT8_C(238), UINT8_C(152), UINT8_C(171) },
      { UINT16_C(45071), UINT16_C(61731), UINT16_C(22550), UINT16_C(38552), UINT16_C(28340), UINT16_C(21974), UINT16_C(14081), UINT16_C(33743) } },
    { { UINT16_C(55344), UINT16_C(64793), UINT16_C(22239), UINT16_C(26428), UINT16_C(29945), UINT16_C(1297), UINT16_C(63634), UINT16_C(25808) },
      { UINT8_C(68), UINT8_C(230), UINT8_C(47), UINT8_C(106), UINT8_C(197), UINT8_C(220), UINT8_C(115), UINT8_C(241),
        UINT8_C(49), UINT8_C(9), UINT8_C(120), UINT8_C(237), UINT8_C(254), UINT8_C(59), UINT8_C(71), UINT8_C(181) },
      { UINT8_C(217), UINT8_C(139), UINT8_C(42), UINT8_C(202), UINT8_C(199), UINT8_C(158), UINT8_C(239), UINT8_C(209),
        UINT8_C(84), UINT8_C(18), UINT8_C(124), UINT8_C(109), UINT8_C(247), UINT8_C(168), UINT8_C(192), UINT8_C(60) },
      { UINT16_C(55379), UINT16_C(64802), UINT16_C(22243), UINT16_C(26556), UINT16_C(29952), UINT16_C(1406), UINT16_C(63755), UINT16_C(25929) } },
    { { UINT16_C(41863), UINT16_C(6642), UINT16_C(38692), UINT16_C(32585), UINT16_C(24700), UINT16_C(3802), UINT16_C(7553), UINT16_C(17842) },
      { UINT8_C(253), UINT8_C(99), UINT8_C(11), UINT8_C(124), UINT8_C(27), UINT8_C(148), UINT8_C(242), UINT8_C(94),
        UINT8_C(95), UINT8_C(134), UINT8_C(228), UINT8_C(48), UINT8_C(248), UINT8_C(20), UINT8_C(129), UINT8_C(123) },
      { UINT8_C(89), UINT8_C(1), UINT8_C(16), UINT8_C(105), UINT8_C(150), UINT8_C(24), UINT8_C(245), UINT8_C(53),
        UINT8_C(183), UINT8_C(143), UINT8_C(56), UINT8_C(241), UINT8_C(187), UINT8_C(109), UINT8_C(103), UINT8_C(209) },
      { UINT16_C(41951), UINT16_C(6651), UINT16_C(38864), UINT16_C(32778), UINT16_C(24761), UINT16_C(3891), UINT16_C(7579), UINT16_C(17928) } },
    { { UINT16_C(10149), UINT16_C(11963), UINT16_C(9875), UINT16_C(34112), UINT16_C(15679), UINT16_C(6977), UINT16_C(13520), UINT16_C(53297) },
      { UINT8_C(104), UINT8_C(180), UINT8_C(15), UINT8_C(130), UINT8_C(179), UINT8_C(51), UINT8_C(122), UINT8_C(216),
        UINT8_C(207), UINT8_C(103), UINT8_C(250), UINT8_C(192), UINT8_C(70), UINT8_C(209), UINT8_C(100), UINT8_C(170) },
      { UINT8_C(121), UINT8_C(12), UINT8_C(116), UINT8_C(114), UINT8_C(198), UINT8_C(162), UINT8_C(89), UINT8_C(79),
        UINT8_C(121), UINT8_C(155), UINT8_C(154), UINT8_C(134), UINT8_C(254), UINT8_C(120), UINT8_C(202), UINT8_C(13) },
      { UINT16_C(10235), UINT16_C(12015), UINT16_C(9971), UINT16_C(34170), UINT16_C(15863), UINT16_C(7066), UINT16_C(13622), UINT16_C(53454) } },
    { { UINT16_C(6132), UINT16_C(31154), UINT16_C(19323), UINT16_C(30639), UINT16_C(16383), UINT16_C(35340), UINT16_C(59095), UINT16_C(40893) },
      { UINT8_C(244), UINT8_C(209), UINT8_C(119), UINT8_C(37), UINT8_C(120), UINT8_C(243), UINT8_C(211), UINT8_C(149),
        UINT8_C(228), UINT8_C(12), UINT8_C(80), UINT8_C(207), UINT8_C(62), UINT8_C(235), UINT8_C(198), UINT8_C(132) },
      { UINT8_C(193), UINT8_C(206), UINT8_C(53), UINT8_C(55), UINT8_C(83), UINT8_C(195), UINT8_C(13), UINT8_C(119),
        UINT8_C(26), UINT8_C(137), UINT8_C(196), UINT8_C(31), UINT8_C(197), UINT8_C(204), UINT8_C(248), UINT8_C(216) },
      { UINT16_C(6334), UINT16_C(31279), UINT16_C(19439), UINT16_C(30815), UINT16_C(16518), UINT16_C(35371), UINT16_C(59145), UINT16_C(40977) } },
    { { UINT16_C(50996), UINT16_C(28664), UINT16_C(52064), UINT16_C(19134), UINT16_C(21713), UINT16_C(53912), UINT16_C(4270), UINT16_C(50615) },
      { UINT8_C(230), UINT8_C(123), UINT8_C(149), UINT8_C(44), UINT8_C(186), UINT8_C(199), UINT8_C(103), UINT8_C(192),
        UINT8_C(183), UINT8_C(132), UINT8_C(167), UINT8_C(198), UINT8_C(201), UINT8_C(208), UINT8_C(173), UINT8_C(39) },
      { UINT8_C(143), UINT8_C(130), UINT8_C(94), UINT8_C(50), UINT8_C(151), UINT8_C(238), UINT8_C(186), UINT8_C(198),
        UINT8_C(196), UINT8_C(174), UINT8_C(44), UINT8_C(25), UINT8_C(23), UINT8_C(53), UINT8_C(237), UINT8_C(205) },
      { UINT16_C(51009), UINT16_C(28706), UINT16_C(52187), UINT16_C(19307), UINT16_C(21891), UINT16_C(54067), UINT16_C(4334), UINT16_C(50781) } },
    { { UINT16_C(12283), UINT16_C(3982), UINT16_C(34342), UINT16_C(51198), UINT16_C(55355), UINT16_C(56804), UINT16_C(15883), UINT16_C(62019) },
      { UINT8_C(226), UINT8_C(164), UINT8_C(115), UINT8_C(124), UINT8_C(69), UINT8_C(123), UINT8_C(127), UINT8_C(253),
        UINT8_C(253), UINT8_C(246), UINT8_C(8), UINT8_C(45), UINT8_C(80), UINT8_C(7), UINT8_C(105), UINT8_C(99) },
      { UINT8_C(246), UINT8_C(12), UINT8_C(48), UINT8_C(245), UINT8_C(253), UINT8_C(204), UINT8_C(22), UINT8_C(52),
        UINT8_C(159), UINT8_C(190), UINT8_C(38), UINT8_C(217), UINT8_C(235), UINT8_C(157), UINT8_C(210), UINT8_C(216) },
      { UINT16_C(12377), UINT16_C(4038), UINT16_C(34372), UINT16_C(51370), UINT16_C(55510), UINT16_C(56954), UINT16_C(15988), UINT16_C(62136) } },
    { { UINT16_C(61236), UINT16_C(244), UINT16_C(60163), UINT16_C(46101), UINT16_C(62573), UINT16_C(55461), UINT16_C(53515), UINT16_C(49904) },
      { UINT8_C(245), UINT8_C(137), UINT8_C(119), UINT8_C(134), UINT8_C(62), UINT8_C(113), UINT8_C(252), UINT8_C(228),
        UINT8_C(242), UINT8_C(150), UINT8_C(233), UINT8_C(138), UINT8_C(70), UINT8_C(47), UINT8_C(205), UINT8_C(181) },
      { UINT8_C(27), UINT8_C(86), UINT8_C(117), UINT8_C(243), UINT8_C(153), UINT8_C(37), UINT8_C(32), UINT8_C(64),
        UINT8_C(24), UINT8_C(234), UINT8_C(152), UINT8_C(40), UINT8_C(105), UINT8_C(251), UINT8_C(177), UINT8_C(226) },
      { UINT16_C(61454), UINT16_C(328), UINT16_C(60244), UINT16_C(46199), UINT16_C(62608), UINT16_C(55665), UINT16_C(53543), UINT16_C(49949) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint16x8_t a = simde_vld1q_u16(test_vec[i].a);
    simde_uint8x16_t b = simde_vld1q_u8(test_vec[i].b);
    simde_uint8x16_t c = simde_vld1q_u8(test_vec[i].c);
    simde_uint16x8_t r = simde_vabal_high_u8(a, b, c);

    simde_test_arm_neon_assert_equal_u16x8(r, simde_vld1q_u16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_vabal_high_u16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    uint32_t a[4];
    uint16_t b[8];
    uint16_t c[8];
    uint32_t r[4];
  } test_vec[] = {
    { { UINT32_C(4090475893), UINT32_C(1066812000), UINT32_C(1297469008), UINT32_C(1097358955) },
      { UINT16_C(17093), UINT16_C(38664), UINT16_C(36237), UINT16_C(33187), UINT16_C(32319), UINT16_C(11451), UINT16_C(23995), UINT16_C(8163) },
      { UINT16_C(63374), UINT16_C(21882), UINT16_C(59381), UINT16_C(39509), UINT16_C(52230), UINT16_C(38063), UINT16_C(38488), UINT16_C(41691) },
      { UINT32_C(4090495804), UINT32_C(1066838612), UINT32_C(1297483501), UINT32_C(1097392483) } },
    { { UINT32_C(3668143220), UINT32_C(837778028), UINT32_C(2568867907), UINT32_C(729070020) },
      { UINT16_C(45026), UINT16_C(15729), UINT16_C(25318), UINT16_C(12934), UINT16_C(51238), UINT16_C(48120), UINT16_C(22733), UINT16_C(65250) },
      { UINT16_C(54573), UINT16_C(28945), UINT16_C(18364), UINT16_C(26218), UINT16_C(43985), UINT16_C(27431), UINT16_C(56699), UINT16_C(11810) },
      { UINT32_C(3668150473), UINT32_C(837798717), UINT32_C(2568901873), UINT32_C(729123460) } },
    { { UINT32_C(3300693276), UINT32_C(1956001735), UINT32_C(2747524670), UINT32_C(4227946335) },
      { UINT16_C(41385), UINT16_C(47036), UINT16_C(37811), UINT16_C(27893), UINT16_C(19015), UINT16_C(25007), UINT16_C(40744), UINT16_C(52349) },
      { UINT16_C(55323), UINT16_C(58978), UINT16_C(48127), UINT16_C(2594), UINT16_C(28665), UINT16_C(34424), UINT16_C(40506), UINT16_C(4345) },
      { UINT32_C(3300702926), UINT32_C(1956011152), UINT32_C(2747524908), UINT32_C(4227994339) } },
    { { UINT32_C(2768463207), UINT32_C(1864318329), UINT32_C(1838739674), UINT32_C(3432191474) },
      { UINT16_C(35878), UINT16_C(57703), UINT16_C(15022), UINT16_C(62453), UINT16_C(6667), UINT16_C(55757), UINT16_C(11030), UINT16_C(17475) },
      { UINT16_C(707), UINT16_C(26315), UINT16_C(34468), UINT16_C(38756), UINT16_C(41507), UINT16_C(11665), UINT16_C(27993), UINT16_C(20345) },
      { UINT32_C(2768498047), UINT32_C(1864362421), UINT32_C(1838756637), UINT32_C(3432194344) } },
    { { UINT32_C(1367580523), UINT32_C(1433377796), UINT32_C(2018433319), UINT32_C(406237960) },
      { UINT16_C(39728), UINT16_C(62750), UINT16_C(33428), UINT16_C(43186), UINT16_C(29864), UINT16_C(46544), UINT16_C(40587), UINT16_C(64161) },
      { UINT16_C(16852), UINT16_C(14073), UINT16_C(2481), UINT16_C(22644), UINT16_C(35281), UINT16_C(36062), UINT16_C(31543), UINT16_C(12317) },
      { UINT32_C(1367585940), UINT32_C(1433388278), UINT32_C(2018442363), UINT32_C(406289804) } },
    { { UINT32_C(535887558), UINT32_C(710011742), UINT32_C(4222389537), UINT32_C(135378958) },
      { UINT16_C(11914), UINT16_C(29816), UINT16_C(13942), UINT16_C(20803), UINT16_C(42772), UINT16_C(22249), UINT16_C(26254), UINT16_C(39349) },
      { UINT16_C(35125), UINT16_C(47647), UINT16_C(20892), UINT16_C(31846), UINT16_C(44144), UINT16_C(5008), UINT16_C(16465), UINT16_C(13907) },
      { UINT32_C(535888930), UINT32_C(710028983), UINT32_C(4222399326), UINT32_C(135404400) } },
    { { UINT32_C(1614503899), UINT32_C(1097701245), UINT32_C(3094653303), UINT32_C(848163008) },
      { UINT16_C(64585), UINT16_C(16113), UINT16_C(36513), UINT16_C(33589), UINT16_C(41213), UINT16_C(59878), UINT16_C(3781), UINT16_C(20408) },
      { UINT16_C(50935), UINT16_C(37934), UINT16_C(636), UINT16_C(36665), UINT16_C(33898), UINT16_C(48284), UINT16_C(7063), UINT16_C(45712) },
      { UINT32_C(1614511214), UINT32_C(1097712839), UINT32_C(3094656585), UINT32_C(848188312) } },
    { { UINT32_C(3669890396), UINT32_C(3210809506), UINT32_C(1506732486), UINT32_C(3654965237) },
      { UINT16_C(13159), UINT16_C(10302), UINT16_C(11123), UINT16_C(24510), UINT16_C(57262), UINT16_C(12154), UINT16_C(28736), UINT16_C(14952) },
      { UINT16_C(54972), UINT16_C(1380), UINT16_C(37704), UINT16_C(9622), UINT16_C(22294), UINT16_C(44461), UINT16_C(8278), UINT16_C(18143) },
      { UINT32_C(3669925364), UINT32_C(3210841813), UINT32_C(1506752944), UINT32_C(3654968428) } },
    { { UINT32_C(3702726748), UINT32_C(1812028967), UINT32_C(2443954047), UINT32_C(2042439423) },
      { UINT16_C(57236), UINT16_C(65112), UINT16_C(35778), UINT16_C(47229), UINT16_C(1331), UINT16_C(1773), UINT16_C(22220), UINT16_C(64020) },
      { UINT16_C(64502), UINT16_C(1794), UINT16_C(41140), UINT16_C(9692), UINT16_C(63262), UINT16_C(3177), UINT16_C(1880), UINT16_C(33419) },
      { UINT32_C(3702788679), UINT32_C(1812030371), UINT32_C(2443974387), UINT32_C(2042470024) } },
    { { UINT32_C(4151596664), UINT32_C(2987569361), UINT32_C(3519196835), UINT32_C(110598850) },
      { UINT16_C(32606), UINT16_C(40627), UINT16_C(19309), UINT16_C(19390), UINT16_C(41508), UINT16_C(39509), UINT16_C(35838), UINT16_C(48754) },
      { UINT16_C(22932), UINT16_C(34644), UINT16_C(52343), UINT16_C(7316), UINT16_C(63638), UINT16_C(33535), UINT16_C(16015), UINT16_C(40245) },
      { UINT32_C(4151618794), UINT32_C(2987575335), UINT32_C(3519216658), UINT32_C(110607359) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint32x4_t a = simde_vld1q_u32(test_vec[i].a);
    simde_uint16x8_t b = simde_vld1q_u16(test_vec[i].b);
    simde_uint16x8_t c = simde_vld1q_u16(test_vec[i].c);
    simde_uint32x4_t r = simde_vabal_high_u16(a, b, c);

    simde_test_arm_neon_assert_equal_u32x4(r, simde_vld1q_u32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_vabal_high_u32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    uint64_t a[2];
    uint32_t b[4];
    uint32_t c[4];
    uint64_t r[2];
  } test_vec[] = {
    { { UINT64_C(9149083788126462255), UINT64_C(7494071422769111859) },
      { UINT32_C(2041335781), UINT32_C(1367478240), UINT32_C(2321114513), UINT32_C(2218486805) },
      { UINT32_C(3958228527), UINT32_C(2693854851), UINT32_C(2017434103), UINT32_C(6484540) },
      { UINT64_C(9149083788430142665), UINT64_C(7494071424981114124) } },
    { { UINT64_C(4423053941100656189), UINT64_C(990907171616478125) },
      { UINT32_C(3796362156), UINT32_C(2589156196), UINT32_C(1422143894), UINT32_C(2696813919) },
      { UINT32_C(867892066), UINT32_C(3702295027), UINT32_C(127003855), UINT32_C(2983215031) },
      { UINT64_C(4423053942395796228), UINT64_C(990907171902879237) } },
    { { UINT64_C(7329956104464289737), UINT64_C(72758202737254580) },
      { UINT32_C(4037095348), UINT32_C(2706697874), UINT32_C(3469671657), UINT32_C(127303580) },
      { UINT32_C(4062092470), UINT32_C(120010243), UINT32_C(3745676276), UINT32_C(3592643144) },
      { UINT64_C(7329956104740294356), UINT64_C(72758206202594144) } },
    { { UINT64_C(5206029364576848437), UINT64_C(5475027404289764645) },
      { UINT32_C(1399070209), UINT32_C(3099532744), UINT32_C(227346930), UINT32_C(1264826698) },
      { UINT32_C(1448391272), UINT32_C(463258493), UINT32_C(3154072378), UINT32_C(397004138) },
      { UINT64_C(5206029367503573885), UINT64_C(5475027405157587205) } },
    { { UINT64_C(4323526344265291552), UINT64_C(7260474860496992020) },
      { UINT32_C(645283633), UINT32_C(3657604684), UINT32_C(2091440748), UINT32_C(1590070019) },
      { UINT32_C(285906177), UINT32_C(3523096247), UINT32_C(2690720407), UINT32_C(4020741804) },
      { UINT64_C(4323526344864571211), UINT64_C(7260474862927663805) } },
    { { UINT64_C(1272135283094720421), UINT64_C(6294270365765646379) },
      { UINT32_C(1551131813), UINT32_C(212006481), UINT32_C(4051741288), UINT32_C(3679029671) },
      { UINT32_C(3027198743), UINT32_C(1339664388), UINT32_C(586741341), UINT32_C(1014799411) },
      { UINT64_C(1272135286559720368), UINT64_C(6294270368429876639) } },
    { { UINT64_C(1848326113476440136), UINT64_C(3301212301176947689) },
      { UINT32_C(3038796523), UINT32_C(571428203), UINT32_C(3729735983), UINT32_C(3044120808) },
      { UINT32_C(3747235367), UINT32_C(3338236402), UINT32_C(3213814495), UINT32_C(1562834250) },
      { UINT64_C(1848326113992361624), UINT64_C(3301212302658234247) } },
    { { UINT64_C(4819464941550472272), UINT64_C(769069127010791191) },
      { UINT32_C(1372130262), UINT32_C(3737707021), UINT32_C(2347251705), UINT32_C(1449293230) },
      { UINT32_C(2853048715), UINT32_C(4169309797), UINT32_C(3631039774), UINT32_C(1783795855) },
      { UINT64_C(4819464942834260341), UINT64_C(769069127345293816) } },
    { { UINT64_C(899408302338136502), UINT64_C(4465131751387657149) },
      { UINT32_C(1832530217), UINT32_C(2112373050), UINT32_C(2047246013), UINT32_C(3014076926) },
      { UINT32_C(2731895957), UINT32_C(2190050819), UINT32_C(3673257953), UINT32_C(1089696283) },
      { UINT64_C(899408303964148442), UINT64_C(4465131753312037792) } },
    { { UINT64_C(9143533208000454769), UINT64_C(5199679203301077444) },
      { UINT32_C(2745288759), UINT32_C(3266596383), UINT32_C(2141100959), UINT32_C(850119357) },
      { UINT32_C(3889231510), UINT32_C(3931186667), UINT32_C(3102691051), UINT32_C(2429610128) },
      { UINT64_C(9143533208962044861), UINT64_C(5199679204880568215) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_uint64x2_t a = simde_vld1q_u64(test_vec[i].a);
    simde_uint32x4_t b = simde_vld1q_u32(test_vec[i].b);
    simde_uint32x4_t c = simde_vld1q_u32(test_vec[i].c);
    simde_uint64x2_t r = simde_vabal_high_u32(a, b, c);

    simde_test_arm_neon_assert_equal_u64x2(r, simde_vld1q_u64(test_vec[i].r));
  }

  return 0;
}


SIMDE_TEST_FUNC_LIST_BEGIN
SIMDE_TEST_FUNC_LIST_ENTRY(vabal_high_s8)
SIMDE_TEST_FUNC_LIST_ENTRY(vabal_high_s16)
SIMDE_TEST_FUNC_LIST_ENTRY(vabal_high_s32)
SIMDE_TEST_FUNC_LIST_ENTRY(vabal_high_u8)
SIMDE_TEST_FUNC_LIST_ENTRY(vabal_high_u16)
SIMDE_TEST_FUNC_LIST_ENTRY(vabal_high_u32)
SIMDE_TEST_FUNC_LIST_END

#include "test-neon-footer.h"
