﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/email/SESRequest.h>
#include <aws/email/SES_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SES {
namespace Model {

/**
 * <p>Represents a request to list the configuration sets associated with your
 * Amazon Web Services account. Configuration sets enable you to publish email
 * sending events. For information about using configuration sets, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/monitor-sending-activity.html">Amazon
 * SES Developer Guide</a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/email-2010-12-01/ListConfigurationSetsRequest">AWS
 * API Reference</a></p>
 */
class ListConfigurationSetsRequest : public SESRequest {
 public:
  AWS_SES_API ListConfigurationSetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListConfigurationSets"; }

  AWS_SES_API Aws::String SerializePayload() const override;

 protected:
  AWS_SES_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A token returned from a previous call to <code>ListConfigurationSets</code>
   * to indicate the position of the configuration set in the configuration set
   * list.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListConfigurationSetsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of configuration sets to return.</p>
   */
  inline int GetMaxItems() const { return m_maxItems; }
  inline bool MaxItemsHasBeenSet() const { return m_maxItemsHasBeenSet; }
  inline void SetMaxItems(int value) {
    m_maxItemsHasBeenSet = true;
    m_maxItems = value;
  }
  inline ListConfigurationSetsRequest& WithMaxItems(int value) {
    SetMaxItems(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_nextToken;

  int m_maxItems{0};
  bool m_nextTokenHasBeenSet = false;
  bool m_maxItemsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SES
}  // namespace Aws
