﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/service-quotas/ServiceQuotasRequest.h>
#include <aws/service-quotas/ServiceQuotas_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ServiceQuotas {
namespace Model {

/**
 */
class PutServiceQuotaIncreaseRequestIntoTemplateRequest : public ServiceQuotasRequest {
 public:
  AWS_SERVICEQUOTAS_API PutServiceQuotaIncreaseRequestIntoTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutServiceQuotaIncreaseRequestIntoTemplate"; }

  AWS_SERVICEQUOTAS_API Aws::String SerializePayload() const override;

  AWS_SERVICEQUOTAS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the quota identifier. To find the quota code for a specific quota,
   * use the <a>ListServiceQuotas</a> operation, and look for the
   * <code>QuotaCode</code> response in the output for the quota you want.</p>
   */
  inline const Aws::String& GetQuotaCode() const { return m_quotaCode; }
  inline bool QuotaCodeHasBeenSet() const { return m_quotaCodeHasBeenSet; }
  template <typename QuotaCodeT = Aws::String>
  void SetQuotaCode(QuotaCodeT&& value) {
    m_quotaCodeHasBeenSet = true;
    m_quotaCode = std::forward<QuotaCodeT>(value);
  }
  template <typename QuotaCodeT = Aws::String>
  PutServiceQuotaIncreaseRequestIntoTemplateRequest& WithQuotaCode(QuotaCodeT&& value) {
    SetQuotaCode(std::forward<QuotaCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the service identifier. To find the service code value for an
   * Amazon Web Services service, use the <a>ListServices</a> operation.</p>
   */
  inline const Aws::String& GetServiceCode() const { return m_serviceCode; }
  inline bool ServiceCodeHasBeenSet() const { return m_serviceCodeHasBeenSet; }
  template <typename ServiceCodeT = Aws::String>
  void SetServiceCode(ServiceCodeT&& value) {
    m_serviceCodeHasBeenSet = true;
    m_serviceCode = std::forward<ServiceCodeT>(value);
  }
  template <typename ServiceCodeT = Aws::String>
  PutServiceQuotaIncreaseRequestIntoTemplateRequest& WithServiceCode(ServiceCodeT&& value) {
    SetServiceCode(std::forward<ServiceCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Amazon Web Services Region to which the template applies.</p>
   */
  inline const Aws::String& GetAwsRegion() const { return m_awsRegion; }
  inline bool AwsRegionHasBeenSet() const { return m_awsRegionHasBeenSet; }
  template <typename AwsRegionT = Aws::String>
  void SetAwsRegion(AwsRegionT&& value) {
    m_awsRegionHasBeenSet = true;
    m_awsRegion = std::forward<AwsRegionT>(value);
  }
  template <typename AwsRegionT = Aws::String>
  PutServiceQuotaIncreaseRequestIntoTemplateRequest& WithAwsRegion(AwsRegionT&& value) {
    SetAwsRegion(std::forward<AwsRegionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the new, increased value for the quota.</p>
   */
  inline double GetDesiredValue() const { return m_desiredValue; }
  inline bool DesiredValueHasBeenSet() const { return m_desiredValueHasBeenSet; }
  inline void SetDesiredValue(double value) {
    m_desiredValueHasBeenSet = true;
    m_desiredValue = value;
  }
  inline PutServiceQuotaIncreaseRequestIntoTemplateRequest& WithDesiredValue(double value) {
    SetDesiredValue(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_quotaCode;
  bool m_quotaCodeHasBeenSet = false;

  Aws::String m_serviceCode;
  bool m_serviceCodeHasBeenSet = false;

  Aws::String m_awsRegion;
  bool m_awsRegionHasBeenSet = false;

  double m_desiredValue{0.0};
  bool m_desiredValueHasBeenSet = false;
};

}  // namespace Model
}  // namespace ServiceQuotas
}  // namespace Aws
