﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticmapreduce/EMR_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace EMR {
namespace Model {

/**
 *  <p>Amazon EMR releases 4.x or later.</p>  <p>An optional
 * configuration specification to be used when provisioning cluster instances,
 * which can include configurations for applications and software bundled with
 * Amazon EMR. A configuration consists of a classification, properties, and
 * optional nested configurations. A classification refers to an
 * application-specific configuration file. Properties are the settings you want to
 * change in that file. For more information, see <a
 * href="https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-configure-apps.html">Configuring
 * Applications</a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/Configuration">AWS
 * API Reference</a></p>
 */
class Configuration {
 public:
  AWS_EMR_API Configuration() = default;
  AWS_EMR_API Configuration(Aws::Utils::Json::JsonView jsonValue);
  AWS_EMR_API Configuration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_EMR_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The classification within a configuration.</p>
   */
  inline const Aws::String& GetClassification() const { return m_classification; }
  inline bool ClassificationHasBeenSet() const { return m_classificationHasBeenSet; }
  template <typename ClassificationT = Aws::String>
  void SetClassification(ClassificationT&& value) {
    m_classificationHasBeenSet = true;
    m_classification = std::forward<ClassificationT>(value);
  }
  template <typename ClassificationT = Aws::String>
  Configuration& WithClassification(ClassificationT&& value) {
    SetClassification(std::forward<ClassificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of additional configurations to apply within a configuration
   * object.</p>
   */
  inline const Aws::Vector<Configuration>& GetConfigurations() const { return m_configurations; }
  inline bool ConfigurationsHasBeenSet() const { return m_configurationsHasBeenSet; }
  template <typename ConfigurationsT = Aws::Vector<Configuration>>
  void SetConfigurations(ConfigurationsT&& value) {
    m_configurationsHasBeenSet = true;
    m_configurations = std::forward<ConfigurationsT>(value);
  }
  template <typename ConfigurationsT = Aws::Vector<Configuration>>
  Configuration& WithConfigurations(ConfigurationsT&& value) {
    SetConfigurations(std::forward<ConfigurationsT>(value));
    return *this;
  }
  template <typename ConfigurationsT = Configuration>
  Configuration& AddConfigurations(ConfigurationsT&& value) {
    m_configurationsHasBeenSet = true;
    m_configurations.emplace_back(std::forward<ConfigurationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of properties specified within a configuration classification.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetProperties() const { return m_properties; }
  inline bool PropertiesHasBeenSet() const { return m_propertiesHasBeenSet; }
  template <typename PropertiesT = Aws::Map<Aws::String, Aws::String>>
  void SetProperties(PropertiesT&& value) {
    m_propertiesHasBeenSet = true;
    m_properties = std::forward<PropertiesT>(value);
  }
  template <typename PropertiesT = Aws::Map<Aws::String, Aws::String>>
  Configuration& WithProperties(PropertiesT&& value) {
    SetProperties(std::forward<PropertiesT>(value));
    return *this;
  }
  template <typename PropertiesKeyT = Aws::String, typename PropertiesValueT = Aws::String>
  Configuration& AddProperties(PropertiesKeyT&& key, PropertiesValueT&& value) {
    m_propertiesHasBeenSet = true;
    m_properties.emplace(std::forward<PropertiesKeyT>(key), std::forward<PropertiesValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_classification;

  Aws::Vector<Configuration> m_configurations;

  Aws::Map<Aws::String, Aws::String> m_properties;
  bool m_classificationHasBeenSet = false;
  bool m_configurationsHasBeenSet = false;
  bool m_propertiesHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMR
}  // namespace Aws
