/*
 * Decompiled with CFR 0.152.
 */
package org.freeplane.plugin.ai.chat.history;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.freeplane.core.resources.ResourceController;
import org.freeplane.core.util.LogUtils;
import org.freeplane.plugin.ai.chat.history.ChatTranscriptId;
import org.freeplane.plugin.ai.chat.history.ChatTranscriptRecord;
import org.freeplane.plugin.ai.chat.history.ChatTranscriptStatus;
import org.freeplane.plugin.ai.chat.history.ChatTranscriptSummary;

public class ChatTranscriptStore {
    private static final String TRANSCRIPT_DIRECTORY = "ai-chats";
    private static final String TRANSCRIPT_EXTENSION = ".json.gz";
    private final ObjectMapper objectMapper;
    private final Path rootDirectory;
    private final DateTimeFormatter dateFormatter;

    public ChatTranscriptStore() {
        this(new ObjectMapper(), ChatTranscriptStore.resolveDefaultRoot());
    }

    ChatTranscriptStore(ObjectMapper objectMapper, Path rootDirectory) {
        this.objectMapper = Objects.requireNonNull(objectMapper, "objectMapper");
        this.rootDirectory = Objects.requireNonNull(rootDirectory, "rootDirectory");
        this.dateFormatter = DateTimeFormatter.ISO_LOCAL_DATE;
    }

    public ChatTranscriptId save(ChatTranscriptRecord record, ChatTranscriptId existingId) {
        Path oldPath;
        if (record == null) {
            return existingId;
        }
        long timestamp = System.currentTimeMillis();
        record.setTimestamp(timestamp);
        ChatTranscriptId targetId = this.buildTranscriptId(existingId, timestamp);
        Path targetPath = this.resolvePath(targetId);
        this.ensureParentDirectory(targetPath);
        this.writeRecord(targetPath, record);
        if (existingId != null && !(oldPath = this.resolvePath(existingId)).equals(targetPath)) {
            this.deletePath(oldPath);
        }
        return targetId;
    }

    public List<ChatTranscriptSummary> list() {
        ArrayList<ChatTranscriptSummary> summaries = new ArrayList<ChatTranscriptSummary>();
        if (!Files.exists(this.rootDirectory, new LinkOption[0])) {
            return summaries;
        }
        try {
            Files.walk(this.rootDirectory, 2, FileVisitOption.FOLLOW_LINKS).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(this::isTranscriptFile).forEach(path -> summaries.add(this.readSummary((Path)path)));
        }
        catch (IOException error) {
            LogUtils.severe((Throwable)error);
        }
        summaries.sort(Comparator.comparingLong(ChatTranscriptSummary::getTimestamp).reversed());
        return summaries;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public ChatTranscriptRecord load(ChatTranscriptId id) {
        if (id == null || id.getFileName() == null) {
            return null;
        }
        Path path = this.resolvePath(id);
        if (!Files.exists(path, new LinkOption[0])) {
            return null;
        }
        try (InputStream fileInputStream = Files.newInputStream(path, new OpenOption[0]);){
            ChatTranscriptRecord chatTranscriptRecord;
            try (GZIPInputStream gzipInputStream = new GZIPInputStream(fileInputStream);){
                chatTranscriptRecord = (ChatTranscriptRecord)this.objectMapper.readValue((InputStream)gzipInputStream, ChatTranscriptRecord.class);
            }
            return chatTranscriptRecord;
        }
        catch (IOException error) {
            LogUtils.severe((Throwable)error);
            return null;
        }
    }

    public boolean delete(ChatTranscriptId id) {
        if (id == null || id.getFileName() == null) {
            return false;
        }
        Path path = this.resolvePath(id);
        return this.deletePath(path);
    }

    public ChatTranscriptId rename(ChatTranscriptId id, String displayName) {
        if (id == null) {
            return null;
        }
        ChatTranscriptRecord record = this.load(id);
        if (record == null) {
            return id;
        }
        record.setDisplayName(displayName);
        return this.save(record, id);
    }

    private void writeRecord(Path path, ChatTranscriptRecord record) {
        try (OutputStream fileOutputStream = Files.newOutputStream(path, new OpenOption[0]);
             GZIPOutputStream gzipOutputStream = new GZIPOutputStream(fileOutputStream);){
            this.objectMapper.writeValue((OutputStream)gzipOutputStream, (Object)record);
        }
        catch (IOException error) {
            LogUtils.severe((Throwable)error);
        }
    }

    private ChatTranscriptSummary readSummary(Path path) {
        ChatTranscriptSummary summary = new ChatTranscriptSummary();
        summary.setId(new ChatTranscriptId(this.rootDirectory.relativize(path).toString()));
        summary.setDisplayName(path.getFileName().toString());
        summary.setStatus(ChatTranscriptStatus.ERROR);
        summary.setTimestamp(this.readLastModifiedTimestamp(path));
        try (InputStream fileInputStream = Files.newInputStream(path, new OpenOption[0]);
             GZIPInputStream gzipInputStream = new GZIPInputStream(fileInputStream);){
            ChatTranscriptRecord record = (ChatTranscriptRecord)this.objectMapper.readValue((InputStream)gzipInputStream, ChatTranscriptRecord.class);
            summary.setStatus(ChatTranscriptStatus.TRANSCRIPT);
            summary.setTimestamp(record.getTimestamp());
            summary.setDisplayName(record.getDisplayName());
            summary.setMapRootShortTextCounts(record.getMapRootShortTextCounts());
        }
        catch (IOException error) {
            summary.setErrorMessage(error.getClass().getSimpleName());
        }
        return summary;
    }

    private long readLastModifiedTimestamp(Path path) {
        try {
            return Files.getLastModifiedTime(path, new LinkOption[0]).toMillis();
        }
        catch (IOException error) {
            return 0L;
        }
    }

    private boolean isTranscriptFile(Path path) {
        String fileName = path.getFileName().toString();
        return fileName.endsWith(TRANSCRIPT_EXTENSION);
    }

    private ChatTranscriptId buildTranscriptId(ChatTranscriptId existingId, long timestamp) {
        Object fileName;
        Object object = fileName = existingId == null ? null : existingId.getLeafFileName();
        if (fileName == null || ((String)fileName).isEmpty()) {
            fileName = UUID.randomUUID().toString() + TRANSCRIPT_EXTENSION;
        }
        String dateFolder = this.dateFormatter.format(LocalDate.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.systemDefault()));
        Path relativePath = Paths.get(dateFolder, new String[0]).resolve((String)fileName);
        return new ChatTranscriptId(relativePath.toString());
    }

    private Path resolvePath(ChatTranscriptId id) {
        return this.rootDirectory.resolve(id.getFileName());
    }

    private void ensureParentDirectory(Path path) {
        Path parent = path.getParent();
        if (parent == null) {
            return;
        }
        try {
            Files.createDirectories(parent, new FileAttribute[0]);
        }
        catch (IOException error) {
            LogUtils.severe((Throwable)error);
        }
    }

    private boolean deletePath(Path path) {
        try {
            return Files.deleteIfExists(path);
        }
        catch (IOException error) {
            LogUtils.severe((Throwable)error);
            return false;
        }
    }

    private static Path resolveDefaultRoot() {
        String userDirectory = ResourceController.getResourceController().getFreeplaneUserDirectory();
        return Paths.get(userDirectory, new String[0]).resolve(TRANSCRIPT_DIRECTORY);
    }
}

