/*
 * Decompiled with CFR 0.152.
 */
package ghidra.pcode.floatformat;

import ghidra.pcode.floatformat.BigFloat;
import ghidra.pcode.floatformat.FloatKind;
import ghidra.pcode.floatformat.UnsupportedFloatFormatException;
import ghidra.pcode.utils.Utils;
import ghidra.util.SystemUtilities;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.Objects;
import javax.help.UnsupportedOperationException;

public class FloatFormat {
    static final FloatFormat JAVA_FLOAT_FORMAT = new FloatFormat(4);
    static final FloatFormat JAVA_DOUBLE_FORMAT = new FloatFormat(8);
    private final int size;
    private final int signbit_pos;
    private final int frac_pos;
    private final int frac_size;
    private final int effective_frac_size;
    private final int exp_pos;
    private final int exp_size;
    private final int bias;
    private final int maxexponent;
    private final boolean jbitimplied;
    public final BigFloat maxValue;
    public final BigFloat minValue;
    private final MathContext displayContext;

    public int getSize() {
        return this.size;
    }

    FloatFormat(int sz) throws UnsupportedFloatFormatException {
        this.size = sz;
        if (this.size == 2) {
            this.signbit_pos = 15;
            this.exp_pos = 10;
            this.exp_size = 5;
            this.frac_pos = 0;
            this.frac_size = 10;
            this.bias = 15;
            this.jbitimplied = true;
            this.displayContext = new MathContext(4, RoundingMode.HALF_EVEN);
        } else if (this.size == 4) {
            this.signbit_pos = 31;
            this.exp_pos = 23;
            this.exp_size = 8;
            this.frac_pos = 0;
            this.frac_size = 23;
            this.bias = 127;
            this.jbitimplied = true;
            this.displayContext = new MathContext(8, RoundingMode.HALF_EVEN);
        } else if (this.size == 8) {
            this.signbit_pos = 63;
            this.exp_pos = 52;
            this.exp_size = 11;
            this.frac_pos = 0;
            this.frac_size = 52;
            this.bias = 1023;
            this.jbitimplied = true;
            this.displayContext = new MathContext(16, RoundingMode.HALF_EVEN);
        } else if (this.size == 16) {
            this.signbit_pos = 127;
            this.exp_pos = 112;
            this.exp_size = 15;
            this.frac_pos = 0;
            this.frac_size = 112;
            this.bias = 16383;
            this.jbitimplied = true;
            this.displayContext = new MathContext(34, RoundingMode.HALF_EVEN);
        } else if (this.size == 32) {
            this.signbit_pos = 255;
            this.exp_pos = 236;
            this.exp_size = 19;
            this.frac_pos = 0;
            this.frac_size = 236;
            this.bias = 262143;
            this.jbitimplied = true;
            this.displayContext = new MathContext(71, RoundingMode.HALF_EVEN);
        } else if (this.size == 10) {
            this.signbit_pos = 79;
            this.exp_pos = 64;
            this.exp_size = 15;
            this.frac_pos = 0;
            this.frac_size = 64;
            this.bias = 16383;
            this.jbitimplied = false;
            this.displayContext = new MathContext(18, RoundingMode.HALF_EVEN);
        } else {
            throw new UnsupportedFloatFormatException(sz);
        }
        if (!this.jbitimplied && this.size <= 8) {
            throw new IllegalArgumentException("Small format implementation assumes jbitimplied=true");
        }
        this.effective_frac_size = this.frac_size + (this.jbitimplied ? 1 : 0);
        this.maxexponent = (1 << this.exp_size) - 1;
        this.maxValue = new BigFloat(this.effective_frac_size, this.exp_size, FloatKind.FINITE, 1, BigInteger.ONE.shiftLeft(this.effective_frac_size).subtract(BigInteger.ONE), (1 << this.exp_size - 1) - 1);
        this.minValue = new BigFloat(this.effective_frac_size, this.exp_size, FloatKind.FINITE, 1, BigInteger.ONE, 2 - (1 << this.exp_size - 1));
    }

    MathContext getDisplayContext() {
        return this.displayContext;
    }

    public BigFloat getMaxBigFloat() {
        return this.maxValue;
    }

    public BigFloat getMinBigFloat() {
        return this.minValue;
    }

    FloatKind extractKind(long encoding) {
        int exp = this.extractExponentCode(encoding);
        if (exp == this.maxexponent) {
            long frac = this.extractFractionalCode(encoding);
            if (frac == 0L) {
                return FloatKind.INFINITE;
            }
            if (frac >>> this.frac_size - 1 == 1L) {
                return FloatKind.QUIET_NAN;
            }
            return FloatKind.SIGNALING_NAN;
        }
        return FloatKind.FINITE;
    }

    private FloatKind extractKind(BigInteger l) {
        int exp = this.extractExponentCode(l);
        if (exp == this.maxexponent) {
            BigInteger frac = this.extractFractionalCode(l);
            if (BigInteger.ZERO.equals(frac)) {
                return FloatKind.INFINITE;
            }
            if (BigInteger.ONE.equals(frac.shiftRight(this.frac_size - 1))) {
                return FloatKind.QUIET_NAN;
            }
            return FloatKind.SIGNALING_NAN;
        }
        return FloatKind.FINITE;
    }

    private long extractFractionalCode(long x) {
        long mask = (1L << this.frac_size) - 1L;
        return (x >>>= this.frac_pos) & mask;
    }

    private BigInteger extractFractionalCode(BigInteger x) {
        BigInteger mask = BigInteger.ONE.shiftLeft(this.frac_size).subtract(BigInteger.ONE);
        return x.shiftRight(this.frac_pos).and(mask);
    }

    private boolean extractSign(long x) {
        return ((x >>>= this.signbit_pos) & 1L) != 0L;
    }

    private boolean extractSign(BigInteger x) {
        return x.testBit(this.signbit_pos);
    }

    private int extractExponentCode(long x) {
        long mask = (1L << this.exp_size) - 1L;
        return (int)((x >>>= this.exp_pos) & mask);
    }

    private int extractExponentCode(BigInteger x) {
        return x.shiftRight(this.exp_pos).intValue() & this.maxexponent;
    }

    private long setSign(long x, boolean sign) {
        if (!sign) {
            return x;
        }
        long mask = 1L;
        return x |= (mask <<= this.signbit_pos);
    }

    private BigInteger setSign(BigInteger x, boolean sign) {
        if (sign) {
            return x.setBit(this.signbit_pos);
        }
        return x;
    }

    public long getZeroEncoding(boolean sgn) {
        return this.setSign(0L, sgn);
    }

    public long getInfinityEncoding(boolean sgn) {
        long res = (long)this.maxexponent << this.exp_pos;
        return this.setSign(res, sgn);
    }

    public BigInteger getBigZeroEncoding(boolean sgn) {
        BigInteger res = BigInteger.ZERO;
        return this.setSign(res, sgn);
    }

    public BigFloat getBigZero(boolean sgn) {
        return new BigFloat(this.effective_frac_size, this.exp_size, FloatKind.FINITE, sgn ? -1 : 1, BigInteger.ZERO, 2 - (1 << this.exp_size - 1));
    }

    public BigInteger getBigInfinityEncoding(boolean sgn) {
        BigInteger res = BigInteger.valueOf(this.maxexponent).shiftLeft(this.exp_pos);
        return this.setSign(res, sgn);
    }

    public BigFloat getBigInfinity(boolean sgn) {
        return BigFloat.infinity(this.effective_frac_size, this.exp_size, sgn ? -1 : 1);
    }

    public long getNaNEncoding(boolean sgn) {
        long res = 1L << this.frac_pos + this.frac_size - 1;
        return this.setSign(res |= (long)this.maxexponent << this.exp_pos, sgn);
    }

    public BigInteger getBigNaNEncoding(boolean sgn) {
        BigInteger res = BigInteger.ONE.shiftLeft(this.frac_pos + this.frac_size - 1);
        res = res.or(BigInteger.valueOf(this.maxexponent).shiftLeft(this.exp_pos));
        return this.setSign(res, sgn);
    }

    public BigFloat getBigNaN(boolean sgn) {
        return BigFloat.quietNaN(this.effective_frac_size, this.exp_size, sgn ? -1 : 1);
    }

    public BigFloat getBigFloat(float f) {
        BigFloat bf = FloatFormat.toBigFloat(f);
        return new BigFloat(this.effective_frac_size, this.exp_size, bf.kind, bf.sign, bf.unscaled.shiftLeft(this.effective_frac_size - bf.fracbits), bf.scale);
    }

    public BigFloat getBigFloat(double d) {
        BigFloat bf = FloatFormat.toBigFloat(d);
        return new BigFloat(this.effective_frac_size, this.exp_size, bf.kind, bf.sign, bf.unscaled.shiftLeft(this.effective_frac_size - bf.fracbits), bf.scale);
    }

    public BigFloat decodeBigFloat(long encoding) {
        int scale;
        if (this.size > 8) {
            throw new UnsupportedOperationException("method not supported for float size of " + this.size);
        }
        boolean sgn = this.extractSign(encoding);
        int exp = this.extractExponentCode(encoding);
        long frac = this.extractFractionalCode(encoding);
        FloatKind kind = this.extractKind(encoding);
        BigInteger unscaled = BigInteger.valueOf(frac);
        if (kind == FloatKind.FINITE) {
            if (exp == 0) {
                scale = -this.bias + 1;
            } else {
                scale = exp - this.bias;
                if (this.jbitimplied) {
                    unscaled = unscaled.setBit(this.frac_size);
                }
            }
        } else {
            scale = 0;
        }
        return new BigFloat(this.effective_frac_size, this.exp_size, kind, sgn ? -1 : 1, unscaled, scale);
    }

    SmallFloatData getSmallFloatData(long encoding) {
        int scale;
        if (this.size > 8) {
            throw new UnsupportedOperationException("method not supported for float size of " + this.size);
        }
        boolean sgn = this.extractSign(encoding);
        int exp = this.extractExponentCode(encoding);
        long frac = this.extractFractionalCode(encoding);
        FloatKind kind = this.extractKind(encoding);
        long unscaled = frac;
        if (kind == FloatKind.FINITE) {
            if (exp == 0) {
                scale = -this.bias + 1;
            } else {
                scale = exp - this.bias;
                if (this.jbitimplied) {
                    unscaled |= 1L << this.frac_size;
                }
            }
        } else {
            scale = 0;
        }
        return new SmallFloatData(this.effective_frac_size, this.exp_size, kind, sgn ? -1 : 1, unscaled, scale);
    }

    public double decodeHostFloat(long encoding) {
        if (this.size == 8) {
            return Double.longBitsToDouble(encoding);
        }
        if (this.size > 8) {
            throw new UnsupportedOperationException("method not supported for float size of " + this.size);
        }
        boolean sgn = this.extractSign(encoding);
        int exp = this.extractExponentCode(encoding);
        long frac = this.extractFractionalCode(encoding);
        boolean subnormal = false;
        if (exp == 0) {
            if (frac == 0L) {
                return sgn ? -0.0 : 0.0;
            }
            subnormal = true;
        } else if (exp == this.maxexponent) {
            if (frac == 0L) {
                return sgn ? Double.NEGATIVE_INFINITY : Double.POSITIVE_INFINITY;
            }
            return Double.NaN;
        }
        exp -= this.bias;
        long msbit = 1 << this.frac_size - 1;
        if (!subnormal && !this.jbitimplied) {
            frac &= msbit ^ 0xFFFFFFFFFFFFFFFFL;
            frac <<= 1;
            --exp;
        }
        if (subnormal) {
            for (exp = -this.bias; exp > -1023 && (frac & msbit) == 0L && frac != 0L; frac <<= 1, --exp) {
            }
            if (exp > -1023 && (frac & msbit) != 0L && frac != 0L) {
                frac <<= 1;
            } else {
                --exp;
            }
            frac &= (long)(~(-(1 << this.frac_size)));
        }
        long encodedDouble = (sgn ? Long.MIN_VALUE : 0L) | (long)(exp += 1023) << 52 | (frac <<= 52 - this.frac_size);
        return Double.longBitsToDouble(encodedDouble);
    }

    public BigFloat decodeBigFloat(BigInteger encoding) {
        boolean sgn = this.extractSign(encoding);
        int sign = sgn ? -1 : 1;
        BigInteger frac = this.extractFractionalCode(encoding);
        int exp = this.extractExponentCode(encoding);
        if (exp == 0) {
            if (frac.signum() == 0) {
                return BigFloat.zero(this.effective_frac_size, this.exp_size, sign);
            }
            return new BigFloat(this.effective_frac_size, this.exp_size, FloatKind.FINITE, sign, frac, 1 - this.bias);
        }
        if (exp == this.maxexponent) {
            if (frac.signum() == 0) {
                return BigFloat.infinity(this.effective_frac_size, this.exp_size, sign);
            }
            return BigFloat.quietNaN(this.effective_frac_size, this.exp_size, sign);
        }
        if (this.jbitimplied) {
            frac = frac.setBit(this.frac_size);
        }
        return new BigFloat(this.effective_frac_size, this.exp_size, FloatKind.FINITE, sign, frac, exp - this.bias);
    }

    public long getEncoding(double host) {
        long fraction;
        int exp;
        SmallFloatData value = FloatFormat.getSmallFloatData(host);
        switch (value.kind) {
            case QUIET_NAN: 
            case SIGNALING_NAN: {
                return this.getNaNEncoding(value.sign < 0);
            }
            case INFINITE: {
                return this.getInfinityEncoding(value.sign < 0);
            }
        }
        if (value.isZero()) {
            return this.getZeroEncoding(value.sign < 0);
        }
        int lb_unscaled = FloatFormat.leadBit(value.unscaled);
        if (value.scale - value.fracbits + lb_unscaled >= -this.bias) {
            exp = value.scale - value.fracbits + 1 + lb_unscaled + this.bias;
            fraction = FloatFormat.roundToLeadBit(value.unscaled, this.frac_size);
            if (FloatFormat.leadBit(fraction) > this.frac_size) {
                fraction >>>= 1;
                ++exp;
            }
            if (this.jbitimplied) {
                fraction &= (1L << this.frac_size) - 1L;
            }
        } else {
            if (!this.jbitimplied) {
                return this.getZeroEncoding(value.sign < 0);
            }
            exp = 0;
            int n = value.scale - value.fracbits + lb_unscaled + this.bias + this.frac_size;
            if (n < 0) {
                return this.getZeroEncoding(value.sign < 0);
            }
            fraction = FloatFormat.roundToLeadBit(value.unscaled, n);
        }
        if (exp >= this.maxexponent) {
            return this.getInfinityEncoding(value.sign < 0);
        }
        long result = (long)exp << this.exp_pos | fraction;
        if (value.sign < 0) {
            result |= 1L << this.signbit_pos;
        }
        return result;
    }

    public BigInteger getEncoding(BigFloat value) {
        BigInteger fraction;
        int exp;
        if (value == null) {
            return this.getBigNaNEncoding(false);
        }
        switch (value.kind) {
            case QUIET_NAN: 
            case SIGNALING_NAN: {
                return this.getBigNaNEncoding(false);
            }
            case INFINITE: {
                return this.getBigInfinityEncoding(value.sign < 0);
            }
        }
        if (value.isZero()) {
            return this.getBigZeroEncoding(value.sign < 0);
        }
        int lb_unscaled = FloatFormat.leadBit(value.unscaled);
        if (value.scale - value.fracbits + lb_unscaled >= -this.bias) {
            exp = value.scale - value.fracbits + 1 + lb_unscaled + this.bias;
            int leadBit = this.frac_size - (this.jbitimplied ? 0 : 1);
            fraction = FloatFormat.roundToLeadBit(value.unscaled, leadBit);
            if (FloatFormat.leadBit(fraction) > this.frac_size) {
                fraction = fraction.shiftRight(1);
                ++exp;
            }
            if (this.jbitimplied) {
                fraction = fraction.clearBit(this.frac_size);
            }
        } else {
            if (!this.jbitimplied) {
                return this.getBigZeroEncoding(value.sign < 0);
            }
            exp = 0;
            int n = value.scale - value.fracbits + lb_unscaled + this.bias + this.frac_size;
            if (n < 0) {
                return this.getBigZeroEncoding(value.sign < 0);
            }
            fraction = FloatFormat.roundToLeadBit(value.unscaled, n);
        }
        if (exp >= this.maxexponent) {
            return this.getBigInfinityEncoding(value.sign < 0);
        }
        BigInteger result = BigInteger.valueOf(exp).shiftLeft(this.exp_pos).or(fraction);
        if (value.sign < 0) {
            result = result.setBit(this.signbit_pos);
        }
        return result;
    }

    public BigDecimal round(BigFloat bigFloat) {
        BigDecimal bigDecimal = bigFloat.toBigDecimal();
        if (bigDecimal == null) {
            return null;
        }
        return bigDecimal.round(this.displayContext);
    }

    public String toDecimalString(BigFloat bigFloat) {
        return bigFloat.toString(this, false);
    }

    public String toDecimalString(BigFloat bigFloat, boolean compact) {
        return bigFloat.toString(this, compact);
    }

    private String toBinaryString(long encoding) {
        String s;
        boolean sgn = this.extractSign(encoding);
        int exp = this.extractExponentCode(encoding);
        long frac = this.extractFractionalCode(encoding);
        FloatKind kind = this.extractKind(encoding);
        switch (kind) {
            case INFINITE: {
                if (sgn) {
                    return "-inf";
                }
                return "+inf";
            }
            case QUIET_NAN: {
                return "qNaN";
            }
            case SIGNALING_NAN: {
                return "sNaN";
            }
            case FINITE: {
                break;
            }
            default: {
                throw new AssertionError((Object)"unexpected kind");
            }
        }
        Object binary = Long.toBinaryString(frac);
        binary = "0".repeat(this.frac_size - ((String)binary).length()) + (String)binary;
        binary = ((String)binary).replaceAll("0*$", "");
        if (((String)binary).isEmpty()) {
            binary = "0";
        }
        String string = s = sgn ? "-" : "";
        if (exp == 0) {
            if (frac == 0L) {
                return String.format("%s0b0.0", s);
            }
            return String.format("%s0b0.%s * 2^%d", s, binary, -this.bias + 1);
        }
        return String.format("%s0b1.%s * 2^%d", s, binary, exp - this.bias);
    }

    private String toBinaryString(BigInteger encoding) {
        String s;
        boolean sgn = this.extractSign(encoding);
        int exp = this.extractExponentCode(encoding);
        BigInteger frac = this.extractFractionalCode(encoding);
        FloatKind kind = this.extractKind(encoding);
        switch (kind) {
            case INFINITE: {
                if (sgn) {
                    return "-inf";
                }
                return "+inf";
            }
            case QUIET_NAN: {
                return "qNaN";
            }
            case SIGNALING_NAN: {
                return "sNaN";
            }
            case FINITE: {
                break;
            }
            default: {
                throw new AssertionError((Object)"unexpected kind");
            }
        }
        Object binary = frac.toString(2);
        binary = "0".repeat(this.frac_size - ((String)binary).length()) + (String)binary;
        binary = ((String)binary).replaceAll("0*$", "");
        if (((String)binary).isEmpty()) {
            binary = "0";
        }
        String string = s = sgn ? "-" : "";
        if (exp == 0) {
            if (BigInteger.ZERO.equals(frac)) {
                return String.format("%s0b0.0", s);
            }
            return String.format("%s0b0.%s * 2^%d", s, binary, -this.bias + 1);
        }
        return String.format("%s0b1.%s * 2^%d", s, binary, exp - this.bias);
    }

    public static BigFloat toBigFloat(float f) {
        return JAVA_FLOAT_FORMAT.decodeBigFloat(0xFFFFFFFFL & (long)Float.floatToRawIntBits(f));
    }

    public static BigFloat toBigFloat(double d) {
        return JAVA_DOUBLE_FORMAT.decodeBigFloat(Double.doubleToRawLongBits(d));
    }

    static SmallFloatData getSmallFloatData(double d) {
        return JAVA_DOUBLE_FORMAT.getSmallFloatData(Double.doubleToRawLongBits(d));
    }

    static String toBinaryString(float f) {
        return JAVA_FLOAT_FORMAT.toBinaryString(0xFFFFFFFFL & (long)Float.floatToRawIntBits(f));
    }

    static String toBinaryString(double d) {
        return JAVA_DOUBLE_FORMAT.toBinaryString(Double.doubleToRawLongBits(d));
    }

    String toBinaryString(BigFloat value) {
        return this.toBinaryString(this.getEncoding(value));
    }

    private static int leadBit(BigInteger i) {
        return i.bitLength() - 1;
    }

    private static int leadBit(long l) {
        return 63 - Long.numberOfLeadingZeros(l);
    }

    private static BigInteger roundToLeadBit(BigInteger i, int newLeadBit) {
        int amt = FloatFormat.leadBit(i) - newLeadBit;
        if (amt == 0) {
            return i;
        }
        if (amt < 0) {
            return i.shiftLeft(-amt);
        }
        int midbit = amt - 1;
        boolean midset = i.testBit(midbit);
        boolean eps = i.getLowestSetBit() < midbit;
        i = i.shiftRight(amt);
        boolean odd = i.testBit(0);
        if (midset && (eps || odd)) {
            i = i.add(BigInteger.ONE);
        }
        return i;
    }

    private static long roundToLeadBit(long i, int newLeadBit) {
        boolean odd;
        int amt = FloatFormat.leadBit(i) - newLeadBit;
        if (amt == 0) {
            return i;
        }
        if (amt < 0) {
            return i << -amt;
        }
        long midbitmask = 1L << amt - 1;
        boolean midset = (i & midbitmask) != 0L;
        boolean eps = (midbitmask - 1L & i) != 0L;
        boolean bl = odd = ((i >>>= amt) & 1L) != 0L;
        if (midset && (eps || odd)) {
            ++i;
        }
        return i;
    }

    public long opEqual(long a, long b) {
        double val2;
        double val1 = this.decodeHostFloat(a);
        long res = val1 == (val2 = this.decodeHostFloat(b)) ? 1L : 0L;
        return res;
    }

    public BigInteger opEqual(BigInteger a, BigInteger b) {
        BigFloat fa = this.decodeBigFloat(a);
        BigFloat fb = this.decodeBigFloat(b);
        if (fa.isNaN() || fb.isNaN()) {
            return BigInteger.ZERO;
        }
        BigInteger res = SystemUtilities.isEqual((Object)fa, (Object)fb) ? BigInteger.ONE : BigInteger.ZERO;
        return res;
    }

    public long opNotEqual(long a, long b) {
        double val2;
        double val1 = this.decodeHostFloat(a);
        long res = val1 != (val2 = this.decodeHostFloat(b)) ? 1L : 0L;
        return res;
    }

    public BigInteger opNotEqual(BigInteger a, BigInteger b) {
        BigFloat fa = this.decodeBigFloat(a);
        BigFloat fb = this.decodeBigFloat(b);
        if (fa.isNaN() | fb.isNaN()) {
            return BigInteger.ONE;
        }
        BigInteger res = SystemUtilities.isEqual((Object)fa, (Object)fb) ? BigInteger.ZERO : BigInteger.ONE;
        return res;
    }

    public long opLess(long a, long b) {
        double val2;
        double val1 = this.decodeHostFloat(a);
        long res = val1 < (val2 = this.decodeHostFloat(b)) ? 1L : 0L;
        return res;
    }

    public BigInteger opLess(BigInteger a, BigInteger b) {
        BigFloat fb;
        BigFloat fa = this.decodeBigFloat(a);
        BigInteger res = fa.compareTo(fb = this.decodeBigFloat(b)) < 0 ? BigInteger.ONE : BigInteger.ZERO;
        return res;
    }

    public long opLessEqual(long a, long b) {
        double val2;
        double val1 = this.decodeHostFloat(a);
        long res = val1 <= (val2 = this.decodeHostFloat(b)) ? 1L : 0L;
        return res;
    }

    public BigInteger opLessEqual(BigInteger a, BigInteger b) {
        BigFloat fb;
        BigFloat fa = this.decodeBigFloat(a);
        BigInteger res = fa.compareTo(fb = this.decodeBigFloat(b)) <= 0 ? BigInteger.ONE : BigInteger.ZERO;
        return res;
    }

    public long opNan(long a) {
        double val = this.decodeHostFloat(a);
        long res = Double.isNaN(val) ? 1L : 0L;
        return res;
    }

    public BigInteger opNan(BigInteger a) {
        BigFloat val = this.decodeBigFloat(a);
        BigInteger res = val.isNaN() ? BigInteger.ONE : BigInteger.ZERO;
        return res;
    }

    public long opAdd(long a, long b) {
        double val1 = this.decodeHostFloat(a);
        double val2 = this.decodeHostFloat(b);
        return this.getEncoding(val1 + val2);
    }

    public BigInteger opAdd(BigInteger a, BigInteger b) {
        BigFloat fa = this.decodeBigFloat(a);
        BigFloat fb = this.decodeBigFloat(b);
        fa.add(fb);
        return this.getEncoding(fa);
    }

    public long opSub(long a, long b) {
        double val1 = this.decodeHostFloat(a);
        double val2 = this.decodeHostFloat(b);
        return this.getEncoding(val1 - val2);
    }

    public BigInteger opSub(BigInteger a, BigInteger b) {
        BigFloat fa = this.decodeBigFloat(a);
        BigFloat fb = this.decodeBigFloat(b);
        fa.sub(fb);
        return this.getEncoding(fa);
    }

    public long opDiv(long a, long b) {
        double val1 = this.decodeHostFloat(a);
        double val2 = this.decodeHostFloat(b);
        return this.getEncoding(val1 / val2);
    }

    public BigInteger opDiv(BigInteger a, BigInteger b) {
        BigFloat fa = this.decodeBigFloat(a);
        BigFloat fb = this.decodeBigFloat(b);
        fa.div(fb);
        return this.getEncoding(fa);
    }

    public long opMult(long a, long b) {
        double val1 = this.decodeHostFloat(a);
        double val2 = this.decodeHostFloat(b);
        return this.getEncoding(val1 * val2);
    }

    public BigInteger opMult(BigInteger a, BigInteger b) {
        BigFloat fa = this.decodeBigFloat(a);
        BigFloat fb = this.decodeBigFloat(b);
        fa.mul(fb);
        return this.getEncoding(fa);
    }

    public long opNeg(long a) {
        double val = this.decodeHostFloat(a);
        return this.getEncoding(-val);
    }

    public BigInteger opNeg(BigInteger a) {
        BigFloat fa = this.decodeBigFloat(a);
        fa.negate();
        return this.getEncoding(fa);
    }

    public long opAbs(long a) {
        double val = this.decodeHostFloat(a);
        return this.getEncoding(Math.abs(val));
    }

    public BigInteger opAbs(BigInteger a) {
        BigFloat fa = this.decodeBigFloat(a);
        fa.abs();
        return this.getEncoding(fa);
    }

    public long opSqrt(long a) {
        double val = this.decodeHostFloat(a);
        return this.getEncoding(Math.sqrt(val));
    }

    public BigInteger opSqrt(BigInteger a) {
        BigFloat fa = this.decodeBigFloat(a);
        fa.sqrt();
        return this.getEncoding(fa);
    }

    public long opInt2Float(long a, int sizein) {
        long ival = a;
        ival = Utils.zzz_sign_extend(ival, 8 * sizein - 1);
        double val = ival;
        return this.getEncoding(val);
    }

    public BigInteger opInt2Float(BigInteger a, int sizein, boolean signed) {
        a = signed ? Utils.convertToSignedValue(a, sizein) : Utils.convertToUnsignedValue(a, sizein);
        return this.getEncoding(this.getBigFloat(a));
    }

    public long opFloat2Float(long a, FloatFormat outformat) {
        double val = this.decodeHostFloat(a);
        return outformat.getEncoding(val);
    }

    public BigInteger opFloat2Float(BigInteger a, FloatFormat outformat) {
        BigFloat fa = this.decodeBigFloat(a);
        return outformat.getEncoding(fa);
    }

    public long opTrunc(long a, int sizeout) {
        double val = this.decodeHostFloat(a);
        long res = (long)val;
        return res &= Utils.calc_mask(sizeout);
    }

    public BigInteger opTrunc(BigInteger a, int sizeout) {
        BigFloat fa = this.decodeBigFloat(a);
        if (fa.isNaN()) {
            return BigInteger.ZERO;
        }
        if (fa.isInfinite()) {
            if (fa.sign > 0) {
                return BigInteger.ONE.shiftLeft(8 * this.size).subtract(BigInteger.ONE).shiftRight(1);
            }
            return BigInteger.ONE.shiftLeft(8 * this.size - 1).negate();
        }
        return fa.toBigInteger();
    }

    public long opCeil(long a) {
        double val = this.decodeHostFloat(a);
        return this.getEncoding(Math.ceil(val));
    }

    public BigInteger opCeil(BigInteger a) {
        BigFloat fa = this.decodeBigFloat(a);
        fa.ceil();
        return this.getEncoding(fa);
    }

    public long opFloor(long a) {
        double val = this.decodeHostFloat(a);
        return this.getEncoding(Math.floor(val));
    }

    public BigInteger opFloor(BigInteger a) {
        BigFloat fa = this.decodeBigFloat(a);
        fa.floor();
        return this.getEncoding(fa);
    }

    public long opRound(long a) {
        double val = this.decodeHostFloat(a);
        return this.getEncoding(Math.floor(val + 0.5));
    }

    public BigInteger opRound(BigInteger a) {
        BigFloat fa = this.decodeBigFloat(a);
        fa.round();
        return this.getEncoding(fa);
    }

    public BigFloat getBigFloat(BigInteger value) {
        if (this.size == 8) {
            double d = value.doubleValue();
            return this.getBigFloat(d);
        }
        if (this.size == 4) {
            float f = value.floatValue();
            return this.getBigFloat(f);
        }
        BigInteger unscaled = value;
        int sign = 1;
        if (unscaled.signum() < 0) {
            sign = -1;
            unscaled = unscaled.negate();
        }
        int scale = this.effective_frac_size - 1;
        int ulen = unscaled.bitLength();
        if (ulen > this.effective_frac_size) {
            int shift = this.effective_frac_size - ulen;
            unscaled = unscaled.shiftLeft(shift);
            scale = this.effective_frac_size - shift - 1;
            if (scale > this.bias) {
                return BigFloat.infinity(this.effective_frac_size, this.exp_size, sign);
            }
        }
        return new BigFloat(this.effective_frac_size, this.exp_size, FloatKind.FINITE, sign, unscaled, scale);
    }

    public BigFloat getBigFloat(String string) throws NumberFormatException {
        Objects.requireNonNull(string);
        if (string.equalsIgnoreCase("NaN")) {
            return BigFloat.quietNaN(this.effective_frac_size, this.exp_size, 1);
        }
        if (string.equalsIgnoreCase("Infinity") || string.equalsIgnoreCase("+Infinity")) {
            return BigFloat.infinity(this.effective_frac_size, this.exp_size, 1);
        }
        if (string.equalsIgnoreCase("-Infinity")) {
            return BigFloat.infinity(this.effective_frac_size, this.exp_size, -1);
        }
        return this.getBigFloat(new BigDecimal(string));
    }

    public BigFloat getBigFloat(BigDecimal value) {
        BigFloat bf;
        if (this.size == 8) {
            return this.getBigFloat(value.doubleValue());
        }
        if (this.size == 4) {
            return this.getBigFloat(value.floatValue());
        }
        BigDecimal val = value;
        if (val.equals(BigDecimal.ZERO)) {
            return BigFloat.zero(this.effective_frac_size, this.exp_size);
        }
        int scale10 = val.scale();
        if (scale10 < 0) {
            scale10 = -scale10;
            BigInteger scalar = BigInteger.valueOf(10L).pow(scale10);
            if ((double)scale10 / 0.3 > (double)this.effective_frac_size) {
                BigInteger intVal = scalar.multiply(val.unscaledValue());
                bf = this.getBigFloat(intVal);
            } else {
                BigFloat scalarBf = this.getBigFloat(scalar);
                bf = this.getBigFloat(val.unscaledValue());
                bf.mul(scalarBf);
            }
        } else if ((double)scale10 / 0.3 >= (double)this.bias) {
            int s1 = scale10 / 2;
            BigInteger bs1 = BigInteger.valueOf(10L).pow(s1);
            BigInteger bs2 = BigInteger.valueOf(10L).pow(scale10 - s1);
            BigFloat bf2 = this.getBigFloat(bs2);
            if (bf2.isInfinite()) {
                return BigFloat.zero(this.effective_frac_size, this.exp_size, value.signum());
            }
            BigFloat bf1 = this.getBigFloat(bs1);
            bf = this.getBigFloat(val.unscaledValue());
            bf.div(bf1);
            bf.div(bf2);
        } else {
            BigInteger scalar = BigInteger.valueOf(10L).pow(scale10);
            BigFloat scalarBf = this.getBigFloat(scalar);
            BigInteger whole = val.unscaledValue();
            bf = this.getBigFloat(whole);
            bf.div(scalarBf);
        }
        return bf;
    }

    static class SmallFloatData {
        final int fracbits;
        final int expbits;
        final FloatKind kind;
        final int sign;
        final long unscaled;
        final int scale;

        public SmallFloatData(int fracbits, int expbits, FloatKind kind, int sign, long unscaled, int scale) {
            this.fracbits = fracbits;
            this.expbits = expbits;
            this.kind = kind;
            this.sign = sign;
            this.unscaled = unscaled;
            this.scale = scale;
        }

        public boolean isZero() {
            return this.kind == FloatKind.FINITE && this.unscaled == 0L;
        }
    }
}

