# (c) Copyright 2009-2012, 2015. CodeWeavers, Inc.

import os
import fcntl
import subprocess
import threading
import time
import collections

import cxproduct
import cxlog
import cxutils
import cxevent
import cxconfig
import cxobjc
import distversion

# for localization
from cxutils import cxgettext as _

class BottleQuery(cxobjc.Proxy):
    pass


class NotFoundError(Exception):
    pass


def _validate_wineprefix(prefix):
    if prefix == "":
        # empty
        return False

    if not os.path.isdir(prefix):
        # not a directory.
        return False

    if not os.access(prefix, os.R_OK):
        # can't read!
        return False

    if not os.path.exists(os.path.join(prefix, "system.reg")):
        return False

    if not os.path.isfile(os.path.join(prefix, "cxbottle.conf")):
        # Probably an orphaned stub.
        return False

    return True


reserved_names = ['default', 'icons', 'installers', 'logs', 'tie']

_bottleListCache = []

def get_bottle_list():
    # pylint: disable=W0603
    global _bottleListCache

    allbottles = set()

    for path in [cxproduct.get_bottle_path(), cxproduct.get_managed_bottle_path()]:
        for dirpath in path.split(":"):
            if not os.path.exists(dirpath):
                continue
            for bottlecandidate in os.listdir(cxutils.string_to_unicode(dirpath)):
                if bottlecandidate in _bottleListCache:
                    allbottles.add(bottlecandidate)
                    continue

                if bottlecandidate.lower() in reserved_names:
                    continue
                if not _validate_wineprefix(os.path.join(dirpath, bottlecandidate)):
                    continue

                allbottles.add(bottlecandidate)

    _bottleListCache = list(allbottles)

    return list(allbottles)


@cxobjc.method(BottleQuery, 'uniqueBottleName_')
def unique_bottle_name(inBasename):
    existingBottles = get_bottle_list()

    collision = True
    candidate = inBasename
    i = 1

    while collision:
        collision = False
        if candidate.lower() in reserved_names:
            collision = True

        if not collision:
            for bottlename in existingBottles:
                if candidate.lower() == bottlename.lower():
                    collision = True
                    break

        if not collision:
            wineprefix = os.path.join(cxproduct.get_bottle_path().split(":")[0], candidate)
            if os.path.exists(wineprefix):
                collision = True

        if collision:
            i += 1
            candidate = inBasename + "-" + str(i)

    return candidate


@cxobjc.method(BottleQuery, 'isValidNewBottleName_')
def is_valid_new_bottle_name(new_bottle_name):
    if not new_bottle_name:
        return False

    if new_bottle_name.lower() in reserved_names:
        return False

    if new_bottle_name != unique_bottle_name(cxutils.sanitize_bottlename(new_bottle_name)):
        return False

    return True


def get_default_bottle():
    for path in [cxproduct.get_bottle_path(), cxproduct.get_managed_bottle_path()]:
        for dirpath in path.split(":"):
            if not os.path.exists(dirpath):
                continue
            defaultcandidate = os.path.join(dirpath, "default")
            if os.path.exists(defaultcandidate):
                return cxutils.string_to_unicode(os.path.basename(os.path.realpath(defaultcandidate)))

    return ""


def get_prefix_for_bottle(bottlename):
    for path in [cxproduct.get_bottle_path(), cxproduct.get_managed_bottle_path()]:
        for dirpath in path.split(":"):
            if not os.path.exists(dirpath):
                continue
            wineprefix = os.path.join(dirpath, bottlename)
            if _validate_wineprefix(wineprefix):
                return wineprefix
    raise NotFoundError("There is no bottle named %s" % repr(bottlename))


_nativePathCache = {}

def get_native_paths(bottlename, windowspaths):

    try:
        bottlePathCache = _nativePathCache[bottlename]
    except KeyError:
        bottlePathCache = {}

    pathsToConvert = []
    for path in windowspaths:
        if path not in bottlePathCache:
            pathsToConvert.append(path)

    if pathsToConvert:
        # Convert those paths which are not in the cache.
        manipulator = BottleManipulator(bottlename)

        if cxlog.is_on('timing'):
            start_time = time.time()

        results = []
        for path in pathsToConvert:
            results.append(manipulator.send('unixpath %s' % path))

        if cxlog.is_on('timing'):
            cxlog.log_('timing', "got %s unix paths in %0.2f seconds" % (' '.join(cxlog.debug_str(x) for x in pathsToConvert), time.time()-start_time))

        for path in pathsToConvert:
            result = results.pop(0).get()
            if result.startswith('path '):
                bottlePathCache[path] = result[5:]
            else:
                # FIXME
                bottlePathCache[path] = ''

    _nativePathCache[bottlename] = bottlePathCache
    # Now our cache has everything we need in it.

    nativePaths = []
    for path in windowspaths:
        nativePaths.append(bottlePathCache[path])

    return nativePaths


def get_native_path(bottlename, windowspath):
    return get_native_paths(bottlename, [windowspath])[0]


_windowsPathCache = {}

def get_windows_paths(bottlename, unixpaths):

    try:
        bottlePathCache = _windowsPathCache[bottlename]
    except KeyError:
        bottlePathCache = {}

    pathsToConvert = []
    for path in unixpaths:
        if path not in bottlePathCache:
            pathsToConvert.append(path)

    if pathsToConvert:
        # Convert those paths which are not in the cache.
        manipulator = BottleManipulator(bottlename)

        if cxlog.is_on('timing'):
            start_time = time.time()

        results = []
        for path in pathsToConvert:
            results.append(manipulator.send('windowspath %s' % path))

        if cxlog.is_on('timing'):
            cxlog.log_('timing', "got %s unix paths in %0.2f seconds" % (' '.join(cxlog.debug_str(x) for x in pathsToConvert), time.time()-start_time))

        for path in pathsToConvert:
            result = results.pop(0).get()
            if result.startswith('path '):
                bottlePathCache[path] = result[5:]
            else:
                # FIXME
                bottlePathCache[path] = ''

    _windowsPathCache[bottlename] = bottlePathCache
    # Now our cache has everything we need in it.

    windowsPaths = []
    for path in unixpaths:
        windowsPaths.append(bottlePathCache[path])

    return windowsPaths


def get_windows_path(bottlename, unixpath):
    return get_windows_paths(bottlename, [unixpath])[0]


def get_control_panel_info(bottlename, windowsdir=None):
    if is_disabled(bottlename):
        return []

    manipulator = BottleManipulator(bottlename)
    if not windowsdir:
        windowsdir = manipulator.send('getwindowsdir').get()

    nativeWindowsDir = get_native_path(bottlename, windowsdir)
    dbFileName = os.path.join(nativeWindowsDir, "control-panel.db")
    iconCacheDir = os.path.join(nativeWindowsDir, "ControlPanelDB")

    try:
        os.mkdir(iconCacheDir)
    except OSError:
        #  Probably the dir already existed -- no problem.
        pass

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, '--no-gui', "--wl-app", "cxcplinfo.exe",
            "--all", dbFileName, iconCacheDir]
    cxutils.run(args, stderr=cxutils.NULL, logprefix=bottlename)

    AppletTable = []
    if os.path.exists(dbFileName):
        cplDB = cxconfig.Raw()
        cplDB.read(dbFileName)
        for applet_id in cplDB:
            if applet_id.lower() == "appwiz.cpl/0":
                # appwiz.cpl is installed because we need it as a back-end.
                #  It's pretty much meaningless in this context, though.
                continue
            if distversion.IS_MACOSX and applet_id.lower() == "desk.cpl/0":
                # The Display Settings control panel is broken on macOS.
                continue
            section = cplDB[applet_id]
            path = section.get('Path', '')
            if path != '':
                AppletTable.append([path, section.get('Name', ''),
                                    section.get('Description', '')])

    # Special case:  Add winecfg
    AppletTable.append(["winecfg.exe", _("Wine Configuration"),
                        _("Built-in tool for configuring bottle settings")])

    # Task Manager...
    AppletTable.append(["taskmgr.exe", _("Task Manager"),
                        _("Manage the applications and processes running in this bottle")])

    # And reboot.
    AppletTable.append(["wineboot.exe", _("Simulate Reboot"),
                        _("Simulate a system reboot in this bottle")])

    if not distversion.IS_MACOSX:
        AppletTable.append(["cxassoceditui", _("Edit Associations"),
                            _("Manage the Windows programs used to open files in the native environment")])
        AppletTable.append(["cxmenueditui", _("Edit Menus"),
                            _("Manage the menus and desktop icons in this bottle")])

    return AppletTable


def config_file_path(bottlename):
    wineprefix = get_prefix_for_bottle(bottlename)
    return os.path.join(wineprefix, "cxbottle.conf")

def get_config(bottlename):
    """Returns a cxconfig.Stack object that takes into account all the
    configuration files relevant to the specified bottle."""
    config = cxproduct.get_config().copy()
    config.addconfig(config_file_path(bottlename))
    return config



def is_disabled(bottlename):
    if distversion.IS_PREVIEW:
        config = get_config(bottlename)
        return config["Bottle"].get("Preview", "0") != "1"

    return False


def is_managed(bottlename):
    config = get_config(bottlename)
    managed = config["Bottle"].get("Updater", "") != ""
    return managed


def get_system_drive(bottlename):
    environ = get_win_environ(bottlename, ("SystemDrive",))
    drive = expand_win_string(environ, "%SystemDrive%")
    return get_native_path(bottlename, drive)


def resolve_icon_path(iconPath, bottlename):
    manipulator = BottleManipulator(bottlename)
    windowspath = manipulator.send('getwindowsdir').get()

    localIconFileName = os.path.join(windowspath, os.path.basename(iconPath))
    if os.path.exists(localIconFileName):
        return localIconFileName
    if os.path.exists(iconPath):
        return iconPath
    nativePath = get_native_path(bottlename, iconPath)
    if os.path.exists(nativePath):
        return nativePath
    return ""


def shutdown_manipulator(bottlename):
    try:
        manipulator = BottleManipulator(bottlename)
        manipulator.clean_shutdown()
    except NotFoundError:
        pass


#####
#
# The RPC implementation
#
#####


class AsyncManipResult(cxevent.AsyncResult):
    def __init__(self, manip):
        cxevent.AsyncResult.__init__(self)
        self.manip = manip

    def _get(self):
        if self.event.isSet():
            return self.poll()
        self.manip.wait(self.event.r_pipe)
        if self.event.isSet():
            return self.poll()

        # wait() returned early because cxmanip quit
        raise IOError("The cxmanip process quit before serving this query.")

    def get(self):
        try:
            return self._get()
        except IOError as e:
            if is_disabled(self.manip.bottlename):
                raise NotFoundError(e).with_traceback(e.__traceback__) from None

            raise

_BottleManipulator_cache = {}

class _BottleManipulator:
    def __init__(self, bottlename):
        self.bottlename = bottlename

        self.lock = threading.Lock()

        self.started = False
        self.got_ok = False
        self.closing = False

        self.thread = self.subp = self.stdin = self.stdin = None
        self.buf = b""

        self.results = []

        # writes to a pipe can block so delay them if necessary
        self.to_write = b''
        self.chars_written = 0

        self._last_activity_time = time.time()

    def _fail_pending_requests(self):
        """self.lock must be acquired when this is called."""

        for result in self.results:
            result.set_error(IOError("The cxmanip process quit before serving this query."))
        self.results = []

        self.to_write = b''
        self.chars_written = 0

    def _start(self):
        """self.lock must be acquired when this is called."""
        if not self.started:
            self._fail_pending_requests()

            winebin = os.path.join(cxutils.CX_ROOT, "bin", "wine")
            args = [winebin, '--no-wait', '--no-gui',
                    '--bottle', self.bottlename, '--wl-app', 'cxmanip']

            cxlog.log("starting command %s in %s" % (' '.join(cxlog.debug_str(x) for x in args), cxlog.to_str(self.bottlename)))

            self.subp = subprocess.Popen(args, stdin=subprocess.PIPE, # pylint: disable=R1732
                                         stdout=subprocess.PIPE,
                                         close_fds=True)
            # use the fd's for stdout and stdin directly to avoid caching
            self.stdin = self.subp.stdin.fileno()
            self.stdout = self.subp.stdout.fileno()

            # make sure called processes will never, ever, ever inherit the pipes
            fcntl.fcntl(self.stdin, fcntl.F_SETFD, fcntl.FD_CLOEXEC)
            fcntl.fcntl(self.stdout, fcntl.F_SETFD, fcntl.FD_CLOEXEC)

            self.buf = b""

            self.started = True
            self.got_ok = False
            self.closing = False

            self.thread = threading.Thread(target=self._thread_proc, args=(self.subp,))
            self.thread.start()

    def _read(self):
        """self.lock must be acquired when this is called
        read from the pipe, without blocking.
        """

        if not self.started:
            return

        readable, _unused, err = cxutils.select([self.stdout], [], [self.stdout], 0)
        if err:
            self.started = False
            cxlog.log("cxmanip process in %s died (read error)" % cxlog.to_str(self.bottlename))
            return
        if readable:
            instr = os.read(self.stdout, 4096)
            if not instr:
                ret = self.subp.poll()
                if ret is not None:
                    cxlog.log("cxmanip process in %s died with return code %d (read error)" % (cxlog.to_str(self.bottlename), ret))
                    self._close()
                return
            self.buf = b'%s%s' % (self.buf, instr)
            self._last_activity_time = time.time()
            while not self.got_ok and b'\n' in self.buf:
                if self.buf.startswith(b"CXMANIP OK\n"):
                    self.got_ok = True
                _dummy, self.buf = self.buf.split(b'\n', 1)
            if self.got_ok:
                while b'\nend\n' in self.buf:
                    this_result = self.results.pop(0)
                    result, self.buf = self.buf.split(b'\nend\n', 1)
                    this_result.set(cxutils.string_to_str(result))
                    if cxlog.is_on('manip'):
                        cxlog.log_('manip', "%s: <- %s" % (cxlog.to_str(self.bottlename), cxlog.to_str(result)))
        if self.closing and not self.results:
            self._close()
            self.closing = False

    def _write(self):
        """self.lock must be acquired when this is called
        write any remaining bytes to the pipe, without blocking.

        Returns True if it worked, False if the process needs to be started.
        """

        if not self.started:
            return False

        while self.chars_written < len(self.to_write):
            _unused, writable, err = cxutils.select([], [self.stdin], [self.stdin], 0)
            if err:
                self.started = False
                cxlog.log("cxmanip process in %s died (write error)" % cxlog.to_str(self.bottlename))
                return False
            if writable:
                self._last_activity_time = time.time()
                try:
                    self.chars_written += os.write(self.stdin, self.to_write[self.chars_written:])
                except OSError:
                    # broken pipe
                    cxlog.log("cxmanip process in %s died (write error)" % cxlog.to_str(self.bottlename))
                    self.started = False
                    return False
            else:
                return True #call would block
        self.chars_written = 0
        self.to_write = b''
        return True

    def wait(self, r=None, timeout=None):
        """blocks until r becomes readable, the cxmanip process dies, or there
        is no activity for at least timeout seconds"""
        while self.started:
            if self.to_write:
                write_sockets = [self.stdin]
                err_sockets = [self.stdin, self.stdout]
            else:
                write_sockets = []
                err_sockets = [self.stdout]
            if r is not None:
                read_sockets = [r, self.stdout]
            else:
                read_sockets = [self.stdout]
            try:
                if timeout is None:
                    readable, writable, err = cxutils.select(read_sockets, write_sockets, err_sockets)
                else:
                    if not self.results and time.time() - self._last_activity_time > timeout:
                        # no activity for timeout seconds
                        return
                    readable, writable, err = cxutils.select(read_sockets, write_sockets, err_sockets, time.time() - self._last_activity_time + timeout + 0.1)
            except cxutils.select_error:
                # one of the sockets has been closed?
                return
            if r is not None and r in readable:
                return
            if err:
                return
            if writable:
                with self.lock:
                    self._write()
            if readable:
                with self.lock:
                    self._read()

    def _thread_proc(self, subp):
        timeout = 15.0 # end the cxmanip process after 15 seconds of inactivity

        while self.started:
            self.wait(r=None, timeout=timeout)

            with self.lock:
                if not self.results and time.time() - self._last_activity_time > timeout:
                    self._close()
                    break
        subp.wait()

    def _send_command(self, string):
        """self.lock must be acquired when this is called.

        Send the given command to cxmanip and return an async result for it,
        if cxmanip is started. Otherwise, return None."""
        result = AsyncManipResult(self)
        self.to_write = b'%s%s\n' % (self.to_write, cxutils.string_to_utf8(string))
        self.results.append(result)
        if self._write():
            return result
        return None

    def send(self, string):
        cxlog.log_('manip', "%s: -> %s" % (cxlog.to_str(self.bottlename), cxlog.debug_str(string)))
        string = cxutils.string_to_utf8(string)
        with self.lock:
            result = self._send_command(string)
            if result is None:
                self._start()
                result = self._send_command(string)
                if result is None:
                    result = AsyncManipResult(self)
                    result.set_error(IOError("Could not start the cxmanip process"))
        return result

    def _close(self):
        """self.lock must be acquired when this is called"""
        if self.started:
            self.subp.stdin.close()
            self.subp.stdout.close()
            self.started = False
            cxlog.log("shutting down cxmanip process in %s" % cxlog.to_str(self.bottlename))

    def close(self):
        with self.lock:
            self._close()

    def clean_shutdown(self):
        """cleanly shut down the cxmanip process when there are no pending requests"""
        with self.lock:
            if not self.results:
                self._close()
            else:
                self.closing = True

    def __del__(self):
        self.close()

def BottleManipulator(bottlename):
    if is_disabled(bottlename):
        raise NotFoundError("Bottle %s is disabled" % repr(bottlename))

    try:
        return _BottleManipulator_cache[bottlename]
    except KeyError:
        _BottleManipulator_cache[bottlename] = _BottleManipulator(bottlename)
        return _BottleManipulator_cache[bottlename]

#####
#
# The remainder of the bottlequery implementation
#
#####

def reset_cache(bottlename):
    if bottlename in _bottleListCache:
        _bottleListCache.remove(bottlename)

    shutdown_manipulator(bottlename)

    _BottleManipulator_cache.pop(bottlename, None)
    _nativePathCache.pop(bottlename, None)
    _windowsPathCache.pop(bottlename, None)


REG_WIN32 = '32\\'
REG_NATIVE = ''
REG_BOTH = '3264\\'

def _manip_unescape(string):
    return string.replace('\\0', '\0').replace('\\n', '\n').replace('\\\\', '\\')

def deserialize_reg(lines):
    subkeys = []
    values = {}
    for line in lines:
        if line.startswith('value '):
            _dummy, rest = line.split(' ', 1)
            valuename, rest = rest.split('\\0 ', 1)
            valuename = _manip_unescape(valuename).rstrip('\0').lower()
            typecode, rest = rest.split(' ', 1)
            if rest.startswith('='):
                # FIXME: multistrings and such not handled properly
                values[valuename] = _manip_unescape(rest[1:]).rstrip('\0')
            else:
                if typecode in ('4', '11'):
                    # little endian integer
                    values[valuename] = int(''.join(rest[x:x+2] for x in range(len(rest)-2, -2, -2)), 16)
                else:
                    # some other kind of data
                    values[valuename] = rest
        elif line.startswith('subkey '):
            _dummy, keyname = line.split(' ', 1)
            keyname = keyname.lower()
            subkeys.append(keyname)
        elif line == '':
            pass
        elif line in ('errno 2', 'error not a valid root key'):
            raise NotFoundError("registry key not found")
        else:
            raise ValueError("got unexpected data: %s" % line)
    return subkeys, values


def get_registry_key(bottlename, path, arch=REG_NATIVE):
    manipulator = BottleManipulator(bottlename)

    cxlog.log("getting registry tree for %s in %s" % (cxlog.debug_str(path), cxlog.to_str(bottlename)))
    if cxlog.is_on('timing'):
        start_time = time.time()

    path = cxutils.string_to_str(path)

    serialdata = manipulator.send('getregkey %s%s' % (arch, path)).get()

    if cxlog.is_on('timing'):
        cxlog.log_('timing', "got registry tree for %s in %0.3f seconds" % (cxlog.debug_str(path), time.time()-start_time))

    return deserialize_reg(serialdata.split('\n'))


def set_registry_key(bottlename, path, name, value, arch=REG_NATIVE):
    manipulator = BottleManipulator(bottlename)

    cxlog.log("setting registry [%s/%s] in %s to %s" % (cxlog.debug_str(path), cxlog.debug_str(name), cxlog.to_str(bottlename), cxlog.debug_str(value)))
    if cxlog.is_on('timing'):
        start_time = time.time()

    path = cxutils.string_to_str(path)
    name = cxutils.string_to_str(name)
    value = cxutils.string_to_str(value)

    command = arch + path
    command = command + "\\n"
    command = command + name
    command = command + "\\n"

    for prefix in ('dword:', 'str:', 'str(2):', 'str(7):', 'hex:'):
        if value.startswith(prefix):
            command = command + value
            break
    else:
        command = command + "str:" + value

    result = manipulator.send('setregkey %s' % command).get()

    if cxlog.is_on('timing'):
        cxlog.log_('timing', "set registry tree for %s in %0.3f seconds" % (cxlog.debug_str(path), time.time()-start_time))

    if result != "success":
        cxlog.log("setting registry error is %s" % (cxlog.to_str(result)))
        return False

    return True

def unset_registry_value(bottlename, path, name, arch=REG_NATIVE):
    manipulator = BottleManipulator(bottlename)

    path = cxutils.string_to_str(path)
    name = cxutils.string_to_str(name)

    command = arch + path
    command = command + "\\n"
    command = command + name

    result = manipulator.send('rmregvalue %s' % command).get()

    if result != "success":
        cxlog.log("removing registry value error is %s" % (cxlog.to_str(result)))
        return False

    return True


WindowsVersion = collections.namedtuple(
    'WindowsVersion',
    ['Version', 'Description', 'CurrentVersion', 'MajorVersion', 'MinorVersion', 'BuildNumber', 'UBR',
     'PlatformId', 'CSDVersion', 'ServicePackMajor', 'ServicePackMinor', 'ProductType'])

# from winecfg/appdefaults.c and ntdll/version.c
_windows_versions = [
    WindowsVersion("win11", "Windows 11", "6.3", 10, 0, 22000, 588, 2, " ", 0, 0, "WinNT"),
    WindowsVersion("win10", "Windows 10", "6.3", 10, 0, 19045, 5796, 2, " ", 0, 0, "WinNT"),
    WindowsVersion("win81", "Windows 8.1", None, 6, 3, 9600, 0, 2, " ", 0, 0, "WinNT"),
    WindowsVersion("win8", "Windows 8", None, 6, 2, 9200, 0, 2, " ", 0, 0, "WinNT"),
    WindowsVersion("win2008r2", "Windows 2008 R2", None, 6, 1, 7601, 0, 2, "Service Pack 1", 1, 0, "ServerNT"),
    WindowsVersion("win7", "Windows 7", None, 6, 1, 7601, 0, 2, "Service Pack 1", 1, 0, "WinNT"),
    WindowsVersion("win2008", "Windows 2008", None, 6, 0, 6002, 0, 2, "Service Pack 1", 0, 0, "ServerNT"),
    WindowsVersion("vista", "Windows Vista", None, 6, 0, 6002, 0, 2, "Service Pack 2", 2, 0, "WinNT"),
    WindowsVersion("win2003", "Windows 2003", None, 5, 2, 3790, 0, 2, "Service Pack 2", 2, 0, "ServerNT"),
    WindowsVersion("winxp", "Windows XP", None, 5, 1, 2600, 0, 2, "Service Pack 3", 3, 0, "WinNT"),
    WindowsVersion("win2k", "Windows 2000", None, 5, 0, 2195, 0, 2, "Service Pack 4", 4, 0, "WinNT"),
    WindowsVersion("winme", "Windows ME", None, 4, 90, 3000, 0, 1, " ", 0, 0, ""),
    WindowsVersion("win98", "Windows 98", None, 4, 10, 2222, 0, 1, " A ", 0, 0, ""),
    WindowsVersion("win95", "Windows 95", None, 4, 0, 950, 0, 1, "", 0, 0, ""),
    WindowsVersion("nt40", "Windows NT 4.0", None, 4, 0, 1381, 0, 2, "Service Pack 6a", 6, 0, "WinNT"),
    WindowsVersion("nt351", "Windows NT 3.5", None, 3, 51, 1057, 0, 2, "Service Pack 2", 0, 0, "WinNT"),
    WindowsVersion("win31", "Windows 3.1", None, 3, 10, 0, 0, 0, "Win32s 1.3", 0, 0, ""),
    WindowsVersion("win30", "Windows 3.0", None, 3, 0, 0, 0, 0, "Win32s 1.3", 0, 0, ""),
    WindowsVersion("win20", "Windows 2.0", None, 2, 0, 0, 0, 0, "Win32s 1.3", 0, 0, ""),
]

_gettable_versions = [
    WindowsVersion("win10", "Windows 10", "6.3", 10, 0, 19043, 1466, 2, " ", 0, 0, "WinNT"),
    WindowsVersion("win10", "Windows 10", "6.3", 10, 0, 18362, 0, 2, " ", 0, 0, "WinNT"),
    WindowsVersion("win10", "Windows 10", None, 10, 0, 10240, 0, 2, " ", 0, 0, "WinNT"),
]
_gettable_versions.extend(_windows_versions)

version_nicknames = {
    "win2000": "win2k",
    "nt2k": "win2k",
    "nt2000": "win2k",
    "win2k3": "win2003",
    "winvista": "vista",
    "win2k8": "win2008",
    "win2k8r2": "win2008r2",
}
_settable_versions = [
    WindowsVersion("winxpsp1", "Windows XP", None, 5, 1, 2600, 0, 2, "Service Pack 1", 1, 0, "WinNT"),
    WindowsVersion("winxpsp2", "Windows XP", None, 5, 1, 2600, 0, 2, "Service Pack 2", 2, 0, "WinNT"),
]
_settable_versions.extend(_windows_versions)

def set_windows_version(bottlename, version):
    version = version_nicknames.get(version, version)

    for info in _settable_versions:
        if info.Version == version:
            break

    config = get_config(bottlename)
    if version == 'winxp' and config["Bottle"].get("WineArch", 'win32') == 'win64':
        # 64-bit winxp version info is different
        info = WindowsVersion("winxp", "Windows XP", None, 5, 2, 3790, 0, 2, "Service Pack 3", 3, 0, "WinNT")

    if info.PlatformId == 0: # VER_PLATFORM_WIN32s
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CSDVersion', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentVersion', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentMajorVersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentMinorVersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentBuildNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentBuild', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'ProductName', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'UBR', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\ProductOptions',
                             'ProductType', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\Windows',
                             'CSDVersion', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\Session Manager\\Environment',
                             'OS', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                             'VersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                             'SubVersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                             'ProductName', REG_BOTH)
        set_registry_key(bottlename, 'HKEY_CURRENT_USER\\Software\\Wine', 'Version', 'str:' + version, REG_BOTH)
    elif info.PlatformId == 1: # VER_PLATFORM_WIN32_WINDOWS
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CSDVersion', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentVersion', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentMajorVersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentMinorVersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentBuildNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'CurrentBuild', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'ProductName', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'UBR', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\ProductOptions',
                             'ProductType', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\Windows',
                             'CSDVersion', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\Session Manager\\Environment',
                             'OS', REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                         'VersionNumber', 'str:%d.%d.%d' % (info.MajorVersion, info.MinorVersion, info.BuildNumber), REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                         'SubVersionNumber', 'str:' + info.CSDVersion, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                         'ProductName', 'str:Microsoft ' + info.Description, REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_CURRENT_USER\\Software\\Wine', 'Version', REG_BOTH)
    elif info.PlatformId == 2: # VER_PLATFORM_WIN32_NT
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'CSDVersion', 'str:' + info.CSDVersion, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'CurrentVersion', 'str:' + (info.CurrentVersion or '%d.%d' % (info.MajorVersion, info.MinorVersion)), REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'CurrentMajorVersionNumber', 'dword:%02x' % info.MajorVersion, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'CurrentMinorVersionNumber', 'dword:%02x' % info.MinorVersion, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'CurrentBuildNumber', 'str:%d' % info.BuildNumber, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'CurrentBuild', 'str:%d' % info.BuildNumber, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                         'ProductName', 'str:Microsoft ' + info.Description, REG_BOTH)
        if info.UBR:
            set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                             'UBR', 'dword:%02x' % info.UBR, REG_BOTH)
        else:
            unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion',
                                 'UBR', REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\ProductOptions',
                         'ProductType', 'str:' + info.ProductType, REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\Windows',
                         'CSDVersion', 'dword:%02x%02x' % (info.ServicePackMajor, info.ServicePackMinor), REG_BOTH)
        set_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\Session Manager\\Environment',
                         'OS', 'Windows_NT', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                             'VersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                             'SubVersionNumber', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion',
                             'ProductName', REG_BOTH)
        unset_registry_value(bottlename, 'HKEY_CURRENT_USER\\Software\\Wine', 'Version', REG_BOTH)

def get_windows_version(bottlename):
    # VER_PLATFORM_WIN32s
    try:
        _subkeys, values = get_registry_key(bottlename, 'HKEY_CURRENT_USER\\Software\\Wine')
        if 'version' in values:
            for info in _gettable_versions:
                if values['version'] == info.Version:
                    return info.Version
    except NotFoundError:
        pass

    # VER_PLATFORM_WIN32_WINDOWS
    try:
        _subkeys, values = get_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows\\CurrentVersion')
        if 'versionnumber' in values:
            for info in _gettable_versions:
                version_number = '%d.%d.%d' % (info.MajorVersion, info.MinorVersion, info.BuildNumber)
                if info.PlatformId == 1 and version_number == values['versionnumber']:
                    return info.Version
    except NotFoundError:
        pass

    # Return winxp if no better match is found
    for best_guess in _windows_versions:
        if best_guess.Version == 'winxp':
            break

    # VER_PLATFORM_WIN32_NT
    try:
        _subkeys, values = get_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\Software\\Microsoft\\Windows NT\\CurrentVersion')
        _subkeys, prod_values = get_registry_key(bottlename, 'HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Control\\ProductOptions')
        if 'currentversion' in values and 'currentbuildnumber' in values and 'producttype' in prod_values:
            for info in _gettable_versions:
                version_number = info.CurrentVersion or '%d.%d' % (info.MajorVersion, info.MinorVersion)
                build_number = '%d' % info.BuildNumber
                if info.PlatformId == 2 and version_number == values['currentversion'] and \
                   info.ProductType == prod_values['producttype']:
                    if build_number == values['currentbuildnumber']:
                        return info.Version

                    if info.BuildNumber > best_guess.BuildNumber and info.BuildNumber < int(values['currentbuildnumber']):
                        best_guess = info
    except NotFoundError:
        pass

    return best_guess.Version


def get_windows_version_name(version):
    version = version_nicknames.get(version, version)

    for info in _gettable_versions:
        if version == info.Version:
            return info.Description

    return _('Unknown')


def get_unix_environ(bottlename=None):
    """Get the Unix environment that should be in effect for the specified
    bottle.

    If bottlename is None, then only the non-bottle specific variables (i.e.
    CX_ROOT) are set."""
    # This should be in BottleWrapper but this would make it impossible to use
    # it from installtask!
    environ = os.environ.copy()
    environ['CX_ROOT'] = cxutils.CX_ROOT
    if bottlename:
        environ['CX_BOTTLE'] = bottlename
        environ['WINEPREFIX'] = get_prefix_for_bottle(bottlename)
    # FIXME: We should probably also take into account the
    # [EnvironmentVariables] section
    # FIXME: We should also cache this data, however this would mean
    # having a mechanism in place to update it.
    return environ

def expand_unix_string(bottlename, string):
    """Expand references to environment variables of the form '${VARNAME}' in
    the specified string.

    The environment is the one appropriate for the specified bottle (see
    get_unix_environ()).
    """
    # This should be in BottleWrapper but this would make it impossible to use
    # it from installtask!
    return cxutils.expand_unix_string(get_unix_environ(bottlename), string)

def get_win_environ(bottlename, varnames=None):
    """Returns a mapping of the specified environment variable names to their
    values. If no variable name is specified, then all the Windows environment
    variables are returned.

    Note that the actual Windows environment is augmented with a couple of
    variables: accessories, desktop, startmenu and programs. In order to get
    these, they must be requested by name though. FIXME: Or not, we'll see.
    """
    environ = {}
    manipulator = BottleManipulator(bottlename)
    for varname in varnames:
        # FIXME: It would be better to retrieve them all at once
        environ[varname.lower()] = manipulator.send('getexpandedstr %s' % varname).get()
    return environ


def expand_win_string(environ, string):
    """Expands references to Windows environment variables using the specified
    environment block.
    """
    while '%' in string:
        try:
            pre, varname, post = string.split('%', 2)
        except ValueError:
            # not enough values to unpack
            break
        varname = varname.lower()
        if varname in environ:
            value = environ[varname]
        else:
            cxlog.warn("unable to expand %s" % cxlog.to_str(varname))
            value = ''
        string = '%s%s%s' % (pre, value, post)
    return string


def add_drive(bottlename, unix_path):
    output = BottleManipulator(bottlename).send('adddrive %s' % unix_path).get()
    output = output.strip()
    if output == '':
        return None
    if output.startswith('drive '):
        return output[6:]
    if output == 'errno 2':
        raise NotFoundError('Path not found %s' % repr(unix_path))
    raise ValueError("got unexpected data: %s" % output)


def rm_drive(bottlename, drive_letter):
    output = BottleManipulator(bottlename).send('rmdrive %s' % drive_letter).get()
    output = output.strip()
    if output == 'errno 2':
        raise NotFoundError('No such drive letter as %s in %s' % (repr(drive_letter), repr(bottlename)))
    if output:
        raise ValueError("got unexpected data: %s" % output)
