# (c) Copyright 2009-2022. CodeWeavers, Inc.

import os
import os.path
import re

import cxfixes
import cxlog
import cxobjc
import cxproduct
import cxutils
import distversion

# for localization
from cxutils import cxgettext as _

import bottlemanagement
import bottlequery
import c4profiles
import c4profilesmanager
import cxaiemedia
import cxdiag
import downloaddetector


#####
#
# Keep track of the installation SourceMedia and of their properties
#
#####

class SourceMedia:
    """Keeps track of installation wizard data associated to a potential
    installation media.
    """

    def __init__(self, path, label, device=''):
        """Source is the absolute path of the installation media. This normally
        corresponds to the mount point of a CD or a directory, but it may also
        point to a user-specified file.

        The label is the string that should be used to describe that media.
        """
        self.path = path
        self.label = label
        self.device = device
        self._profile_ids = None

    def _get_profile_ids(self):
        """Returns the list of profile ids that this media could correspond to
        (in the form of a dictionary mapping the ids to the profile objects).
        """
        if self._profile_ids is None:
            self._profile_ids = c4profilesmanager.get_matching_profiles(self.path)
        return self._profile_ids

    profile_ids = property(_get_profile_ids)



#####
#
# Keep track of the target bottles and of their properties
#
#####

# Bottle categories
CAT_NONE = 0
CAT_RECOMMENDED = 1
CAT_COMPATIBLE = 2
CAT_INCOMPATIBLE = 3

# Reasons for an appid to be included in the install
REASON_MAIN = 0
REASON_DEPENDENCY = 1
REASON_STEAM = 2
REASON_OVERRIDE = 3

# Dependency override types
OVERRIDE_EXCLUDE = 0
OVERRIDE_INCLUDE = 1

class TargetBottle:

    #####
    #
    # Initialization
    #
    #####

    def __init__(self, installtask, bottle):
        self.installtask = installtask
        self.bottle = bottle

        if bottle:
            self.bottlename = bottle.name
        else:
            self.bottlename = None

        self._newtemplate = None

        # The profile-related properties
        # This should match check_cache()
        self._profile = None
        self._locale = None
        self._category = CAT_NONE
        self._installers = None
        self._missing_media = None
        self._missing_profiles = None
        self._warnings = None
        self._use_steam = False
        self._dep_overrides = {}
        self._sourcefile = None
        self._sourcetype = None
        self._reasons = None

        self.compatible = True
        self.is_template = False
        self.installed_packages_ready = False

    def _gettemplate(self):
        if self._newtemplate is None:
            return self.bottle.template
        return self._newtemplate

    template = property(_gettemplate)

    def is64bit(self):
        template = self.template
        return template.endswith('_64')


    #####
    #
    # Profile dependency analysis
    #
    #####

    def _get_sourcetype(self):
        # Possible values:
        #  dependency - this is a dependency of the main profile, so the
        #   sourcetype value should not be used (because the user has no
        #   opportunity to select the source until after the install starts, or
        #   maybe never).
        #  steam - installed from a steam id
        #  file - installed from a local or downloaded file
        #  cd - installed from a cd or other local directory
        #  unset - user has not yet selected an install source
        if self.installtask.installWithSteam:
            return 'steam'
        if self.installtask.installerDownloadSource:
            # We could report this as a "download", but it's probably bad for
            # profiles to behave differently depending on whether you download
            # the file or let CrossOver do it. When adding different
            # distributors (bug 9162), we'll need to be careful to make sure the
            # user can select the distributor for a local file.
            return 'file'
        if self.installtask.installerSource:
            if os.path.isdir(self.installtask.installerSource):
                return 'cd'
            return 'file'
        # We can't ever start an install with this, but if someone wants to
        # set installnotes that apply only to file installs and not to cd
        # installs, this will let them delay the notes until a source is
        # chosen.
        return 'unset'

    def check_cache(self):
        """Checks whether the cached data is still relevant and throws it away
        if not (typically because either the profile or the locale changed).
        """
        # If further optimisation is needed we can check for the presence of
        # useif or even scan its content before resetting the cache for a mere
        # locale change.
        if self._profile != self.installtask.profile or \
                self._locale != self.installtask.realInstallerLocale or \
                self._use_steam != self.installtask.installWithSteam or \
                self._dep_overrides != self.installtask.dependency_overrides or \
                self._sourcefile != self.installtask.installerSource or \
                self._sourcetype != self._get_sourcetype():
            self._profile = self.installtask.profile
            self._use_steam = self.installtask.installWithSteam
            self._dep_overrides = self.installtask.dependency_overrides.copy()
            self._locale = self.installtask.realInstallerLocale
            self._sourcefile = self.installtask.installerSource
            self._sourcetype = self._get_sourcetype()
            # Wipe everything out
            self._category = CAT_NONE
            self._installers = None
            self._missing_media = None
            self._missing_profiles = None
            self._warnings = None
            self._reasons = None

    def analyzed(self):
        """Returns True if the bottle has been analyzed already and False
        otherwise.
        """
        self.check_cache()
        return self._installers is not None

    def _break_dependency_loops(self, appid, parents):
        """Detects, breaks and reports dependency loops."""
        parents.add(appid)
        to_remove = set()
        for depid in self._installers[appid].pre_dependencies:
            if depid in parents:
                if not self.bottlename:
                    cxlog.warn("The dependency of %s on %s creates a loop in bottle template %s." % (cxlog.to_str(self.installtask.profiles[appid].name), cxlog.to_str(self.installtask.profiles[depid].name), self.template))
                to_remove.add(depid)
            elif depid in self._installers:
                self._break_dependency_loops(depid, parents)
            else:
                # This dependency is either already installed, or it has no
                # profile at all. Either way we should ignore it.
                to_remove.add(depid)
        if to_remove:
            self._installers[appid].pre_dependencies -= to_remove
        parents.remove(appid)

    def _add_reason(self, appids, reason, other_appid=None):
        for appid in appids:
            if appid not in self._reasons:
                self._reasons[appid] = (reason, other_appid)

    def _is_predependency(self, dep_appid, appid):
        """Returns True if dep_appid must install before appid, must be called
        after postdependencies are mapped to predependencies."""
        todo = set([appid])
        checked = set()
        while todo:
            check_appid = todo.pop()
            if check_appid in checked or check_appid not in self._installers:
                continue
            checked.add(check_appid)
            if dep_appid in self._installers[check_appid].pre_dependencies:
                return True
            todo.update(self._installers[check_appid].pre_dependencies)
        return False

    def analyze(self):
        """Analyzes the bottle so that we know all there is to know for
        installation and about its suitability for the selected profile and
        locale. This means:
        - creating aggregated installer profiles for the selected profile and
          all its dependencies,
        - detecting and reporting any potential issue like dependency loops,
          template incompatibilities, etc.
        """
        if self.analyzed():
            return True

        if self.bottle:
            if not self.bottle.installed_packages_ready:
                # This bottle's installed applications list is not ready yet
                # so we cannot perform the analysis. So return False so the
                # caller knows he should try again later.
                cxlog.log("analyze: %s is not ready" % cxlog.to_str(self.bottlename))
                return False
            installed_apps = self.bottle.installed_packages
        else:
            installed_apps = {}

        profiles = self.installtask.profiles

        self._installers = {}
        self._missing_media = set()
        self._missing_profiles = set()
        self._warnings = []
        self._reasons = {}
        self.compatible = True

        override_include_deps = set(appid for appid in self._dep_overrides if self._dep_overrides[appid] == OVERRIDE_INCLUDE)

        incompatible_profiles = []
        overrides = {}
        dependencies = {}
        todo = [self._profile.appid]
        self._add_reason((self._profile.appid,), REASON_MAIN)
        while todo or override_include_deps: # pylint: disable=R1702
            if todo:
                appid = todo.pop()
            else:
                appid = override_include_deps.pop()
                self._add_reason((appid,), REASON_OVERRIDE)

            if appid in self._installers or \
               (appid != self._profile.appid and appid in installed_apps):
                # We've done that one already. This check also lets us avoid
                # infinite loops.
                continue

            if appid not in profiles or not profiles[appid].is_for_current_product:
                # This dependency has no profile *at all* (not even a name),
                # or it is for the wrong CrossOver product.
                self._missing_profiles.add(appid)
                continue

            if self._dep_overrides.get(appid) == OVERRIDE_EXCLUDE:
                continue

            profile = profiles[appid]
            if appid == self._profile.appid:
                profile = self._profile

            # Aggregate all the installer profile chunks into one installer
            # profile.
            installer = c4profiles.C4InstallerProfile()
            nomatch = True
            for inst_profile in profiles.installer_profiles(appid):
                if inst_profile.use_if is None or inst_profile.use(self.get_use_if_properties(inst_profile.appid or appid)):
                    if inst_profile.appid is None or inst_profile.appid == appid:
                        installer.update(inst_profile)
                        nomatch = False
                    else:
                        # Remember the override information so we can apply it
                        # after we have built the main installer profile.
                        if inst_profile.appid not in overrides:
                            overrides[inst_profile.appid] = []
                        overrides[inst_profile.appid].append(inst_profile)

                        if inst_profile.appid in self._installers:
                            # We may have just discovered new dependencies for
                            # a profile we already handled.
                            todo.extend(inst_profile.pre_dependencies)
                            todo.extend(inst_profile.post_dependencies)
                            self._add_reason(inst_profile.pre_dependencies, REASON_DEPENDENCY, inst_profile.appid)
                            self._add_reason(inst_profile.post_dependencies, REASON_DEPENDENCY, inst_profile.appid)
                        else:
                            # Remember the extra dependencies for that profile
                            # so we take them into account if we run into it
                            # later on.
                            if inst_profile.appid not in dependencies:
                                dependencies[inst_profile.appid] = set()
                            dependencies[inst_profile.appid].update(inst_profile.pre_dependencies)
                            dependencies[inst_profile.appid].update(inst_profile.post_dependencies)

            if nomatch:
                # There is no installer profile so use the one for the unknown
                # applications.
                installer = profiles.unknown_installer().copy()

            if self._use_steam and appid == self._profile.appid:
                installer.pre_dependencies.add("com.codeweavers.c4.206")
                todo.append("com.codeweavers.c4.206")
                self._add_reason(("com.codeweavers.c4.206",), REASON_STEAM)

            installer.parent = profile
            self._installers[appid] = installer

            # Check that bottle's template is compatible with this installer
            if appid == self._profile.appid:
                purpose = 'use'
            else:
                purpose = 'install'
            if self.template not in profile.app_profile.bottle_types[purpose]:
                incompatible_profiles.append(profile.name)

            if appid != self._profile.appid and \
                   'virtual' not in profile.app_profile.flags and \
                   not cxaiemedia.get_builtin_installer(appid) and \
                   not profile.app_profile.download_urls:
                if profile.app_profile.steamid:
                    # Use steam for dependency with steamid and no other install source
                    installer.pre_dependencies.add("com.codeweavers.c4.206")
                    todo.append("com.codeweavers.c4.206")
                    self._add_reason(("com.codeweavers.c4.206",), REASON_DEPENDENCY, appid)
                else:
                    # We cannot automatically download this dependency.
                    # The installation engine does not care, but still make a note
                    # of it for the GUI.
                    self._missing_media.add(appid)

            # Schedule the dependencies for analysis
            if appid not in dependencies:
                dependencies[appid] = set()
            dependencies[appid].update(installer.pre_dependencies)
            dependencies[appid].update(installer.post_dependencies)

            todo.extend(dependencies[appid])
            self._add_reason(dependencies[appid], REASON_DEPENDENCY, appid)

        if incompatible_profiles:
            apps = ', '.join(sorted(incompatible_profiles))
            if self.template in self._profile.app_profile.bottle_types['use']:
                if not self.bottlename:
                    cxlog.warn("The %s template is compatible with %s but is incompatible with dependencies: %s" % (self.template, cxlog.to_str(self._profile.appid), cxlog.to_str(apps)))
            else:
                self.compatible = False
                if self.bottlename:
                    self._warnings.append(_("You have selected the '%(bottlename)s' bottle, which is incompatible with %(applications)s.") %
                                          {'bottlename': cxutils.html_escape(self.bottlename),
                                           'applications': cxutils.html_escape(apps)})
                else:
                    self._warnings.append(_("You have chosen to install in a new %(template)s bottle but it is incompatible with %(applications)s.") %
                                          {'template': self.template,
                                           'applications': cxutils.html_escape(apps)})

        # Apply all the installer overrides so we don't have to worry about
        # them later.
        for appid, installer_profiles in overrides.items():
            if appid in self._installers:
                for inst_profile in installer_profiles:
                    # Note that this may impact the pre_dependencies list, but
                    # we took precautions in the above loop so it won't
                    # reference new profiles.
                    installer = self._installers[appid]
                    installer.update(inst_profile)

        for appid in self._installers:
            installer = self._installers[appid]
            for postdep_appid in installer.post_dependencies:
                # If we have a post-dependency, we should make sure that it
                # is not installed before this profile.
                if postdep_appid in self._installers:
                    self._installers[postdep_appid].pre_dependencies.add(appid)

        # Make sure that manually added profiles have a defined relationship to
        # the main profile
        for appid in self._installers:
            if self._reasons.get(appid) == (REASON_OVERRIDE, None):
                if self._is_predependency(self._profile.appid, appid) or \
                   self._profile.appid in self._installers[appid].parent.app_profile.extra_fors:
                    self._installers[appid].pre_dependencies.add(self._profile.appid)
                    self._installers[self._profile.appid].post_dependencies.add(appid)
                else:
                    self._installers[self._profile.appid].pre_dependencies.add(appid)

        # Break dependency loops
        self._break_dependency_loops(self._profile.appid, set())

        untrusted_profiles = []
        for appid in self._installers:
            if not self._installers[appid].parent.trusted:
                untrusted_profiles.append(self._installers[appid].parent.name)

        if untrusted_profiles:
            untrusted = ', '.join(sorted(untrusted_profiles))
            self._warnings.append(_("<span style='color: red;'>This installation will include profiles from untrusted sources</span>: %s.") % cxutils.html_escape(untrusted))

        return True

    def _getinstallers(self):
        """Returns the aggregated installer profiles for the application and
        all its dependencies.

        Note: Installation information can be split up across multiple
        C4InstallerProfile objects which makes it unusable as is. This is
        unlike the installer profiles in this map where all the pieces have
        been 'aggregated' together into a single installer profile object.
        """
        self.analyze()
        return self._installers

    installers = property(_getinstallers)

    def _getmissing_dependencies(self):
        """Returns the dependency application ids for which have not been
        installed in the target bottle.
        """
        self.analyze()

        ret = []
        if not self._installers:
            return ret

        for appid, installer in self._installers.items():
            app_profile = installer.parent.app_profile
            if appid != self._profile.appid and (
                   app_profile.installed_key_pattern or
                   app_profile.installed_display_pattern or
                   app_profile.installed_registry_globs or
                   app_profile.installed_file_globs or
                   app_profile.steamid):
                ret.append(appid)

        return ret

    missing_dependencies = property(_getmissing_dependencies)

    def _getmissing_media(self):
        """Returns the dependency application ids for which we cannot
        automatically get an installer. The GUI will have to ask the user for
        an installation source for these.
        """
        self.analyze()
        return self._missing_media

    missing_media = property(_getmissing_media)

    def _getmissing_profiles(self):
        """Returns the dependencies for which there is no C4 profile
        information at all for the current product.
        """
        self.analyze()
        return self._missing_profiles

    missing_profiles = property(_getmissing_profiles)

    def _getwarnings(self):
        """Returns a list of localized strings describing the issues found
        while analyzing this bottle.

        This may be the presence of dependency loops for instance, missing
        profiles, etc.
        """
        self.analyze()
        return self._warnings

    warnings = property(_getwarnings)

    def _getreasons(self):
        """Returns a dictionary mapping appids in the install to the reason
        they will be installed, as a tuple of (code, appid). code is REASON_MAIN
        for the main selected profile, REASON_DEPENDENCY for dependencies, or
        REASON_STEAM for the Steam profile if the install source is Steam.

        If code is REASON_DEPENDENCY, appid is the profile with the dependency."""
        self.analyze()
        return self._reasons

    reasons = property(_getreasons)

    def get_use_if_properties(self, appid):
        """Returns a mapping containing the use_if properties for the selected
        profile, locale, source and bottle combination."""
        self.check_cache()
        if appid == self._profile.appid:
            sourcetype = self._sourcetype
        else:
            sourcetype = 'dependency'
        cxversion_x, cxversion_y, _dummy = (distversion.CX_VERSION + '..').split('.', 2)
        properties = {'product': distversion.BUILTIN_PRODUCT_ID,
                      'cxversion': distversion.CX_VERSION,
                      'cxversion.x': cxversion_x.rjust(2, '0'),
                      'cxversion.y': cxversion_y.rjust(2, '0'),
                      'platform': distversion.PLATFORM,
                      'appid': appid,
                      'locale': self._locale,
                      'bottletemplate': self.template,
                      'sourcetype': sourcetype}
        if appid == self._profile.appid and self._sourcetype == 'file' and \
           self._sourcefile is not None:
            properties['sourcefile'] = os.path.basename(self._sourcefile)
        if cxproduct.is_wow64_install():
            flags = cxdiag.CHECK_64BIT
        else:
            flags = cxdiag.CHECK_32BIT
            if self.is64bit():
                flags |= cxdiag.CHECK_64BIT
        diag = cxdiag.get(self.bottlename, flags)
        for prop, value in diag.properties.items():
            if prop not in properties:
                properties[prop] = value
            else:
                cxlog.err("the %s cxdiag property collides with a builtin one" % prop)
        return properties



    #####
    #
    # Assign a category to the bottle
    #
    #####

    def has_category(self):
        """Returns True if we have determined this bottle's category."""
        self.check_cache()
        return self._category != CAT_NONE

    def _cache_category(self, category):
        """A convenience method that caches the category and returns it."""
        self._category = category
        return self._category

    def get_category(self):
        """Returns the bottle's category which is a measure of its suitability
        for the profile selected for installation.

        CAT_INCOMPATIBLE means the bottle is incompatible with it.
        CAT_RECOMMENDED means we recommend installing in this bottle, which
          happens if it contains an application the profile is an extra for,
          or that belongs in the same application bottle group as the profile.
        CAT_COMPATIBLE means the bottle is compatible.
        """
        self.check_cache()
        if self._category != CAT_NONE or not self._profile:
            return self._category

        if self.template not in self._profile.app_profile.bottle_types['use']:
            # The bottle's template is not in the profile's set of compatible
            # templates. So we need look no further.
            return self._cache_category(CAT_INCOMPATIBLE)

        app_profile = self._profile.app_profile

        app_profile_application_group = app_profile.application_group
        if self._use_steam:
            app_profile_application_group = "Steam"

        if self.bottle and not app_profile.extra_fors and \
                not app_profile_application_group:
            # This is a real bottle, and the profile has no property that would
            # let us recommend a specific bottle. So we may be able to base our
            # analysis on the template alone.
            target_template = self.installtask.templates[self.template]
            if target_template.get_category() == CAT_COMPATIBLE:
                # If the profile is compatible with the template, then it
                # should be compatible with any bottle based on it (because
                # the only source of incompatibilities are the dependencies and
                # new bottles are where we have most of them).
                return self._cache_category(CAT_COMPATIBLE)

        # We need to really analyze the dependencies now.
        if not self.analyze():
            # Return CAT_NONE so the caller knows he should try again later.
            return CAT_NONE

        # If the bottle analysis found issues, then the bottle is incompatible.
        if not self.compatible:
            return self._cache_category(CAT_INCOMPATIBLE)

        # Prefer bottles containing one of the applications we are an extra
        # for, and those containing applications that belong to our
        # application bottle group.
        if app_profile and self.bottle and \
                (app_profile.extra_fors or app_profile_application_group):
            for (appid, installed) in self.bottle.installed_packages.items():
                if appid in app_profile.extra_fors:
                    return self._cache_category(CAT_RECOMMENDED)
                if app_profile_application_group and \
                        installed.profile and installed.profile.app_profile:
                    app_group = installed.profile.app_profile.application_group
                    if app_profile_application_group == app_group:
                        return self._cache_category(CAT_RECOMMENDED)

        return self._cache_category(CAT_COMPATIBLE)

    # Alias for Objective-C
    # TODO: this should be replaced with a namedSelector decorator on get_category,
    # but that fails with 'no such selector: getCategory'. Can't figure out why.
    getCategory = get_category

    def _get_installprofile(self):
        installers = self.installers
        if installers:
            return installers[self._profile.appid]
        return None

    installprofile = property(_get_installprofile)


class TargetTemplate(TargetBottle):
    def __init__(self, installtask, template):
        TargetBottle.__init__(self, installtask, None)
        self._newtemplate = template
        self.is_template = True


#####
#
# InstallTask delegate
#
#####

class InstallTaskDelegate:
    """This class defines the delegate interface that InstallTask uses to
    notify the GUI of changes in the installtask state.

    The GUI (especially the Mac one) does not have to provide an object that
    derives from this class, but it must implement all of the methods defined
    here.
    """

    @cxobjc.delegate
    def profileChanged(self):
        """Notifies the GUI that the profile selected for installation has
        changed.
        """

    @cxobjc.delegate
    def sourceChanged(self):
        """Notifies the GUI that the media selected for the installation has
        changed.
        """

    @cxobjc.delegate
    def categorizedBottle_(self, target_bottle):
        """Notifies the GUI that the specified bottle's category is now
        available, where target_bottle is either a TargetBottle or a
        TargetTemplate object.

        This may be used to implement a GUI that shows the bottles right away
        and then updates a field as their categories become known.
        """

    @cxobjc.delegate
    def categorizedAllBottles(self):
        """Notifies the GUI that all the bottles have been assigned a
        category.

        It also means that the bottle picking, if any, is done.
        This can be used to implement a GUI which only shows list of bottles
        one can install into once all their categories are known.
        """

    @cxobjc.delegate
    def analyzedBottle_(self, target_bottle):
        """Notifies the GUI that the specified bottle has been fully analyzed.

        This means we have determined everything there is to know in order to
        perform the installation into this bottle. So once one has received
        this notification one can query the list of dependencies that will
        need to be installed, and the list issues (such as dependency loops,
        etc).
        Note that the order of the analyzedBottle_() and categorizedBottle_()
        notifications is undefined. However, once a bottle has been analyzed
        its category can be computed and it is ready for installation (if
        compatible).
        """

    @cxobjc.delegate
    def bottleCreateChanged(self):
        """Notifies the GUI that the flag specifying whether to install in a
        new bottle or in an existing one has changed.
        """

    @cxobjc.delegate
    def bottleNewnameChanged(self):
        """Notifies the GUI that the new bottle name has been modified.
        FIXME: Specify if one can receive such notifications when not
        installing in a new bottle.
        """

    @cxobjc.delegate
    def bottleTemplateChanged(self):
        """Notifies the GUI that the new bottle template has been modified.
        FIXME: Specify if one can receive such notifications when not
        installing in a new bottle.
        """

    @cxobjc.delegate
    def bottleNameChanged(self):
        """Notifies the GUI that the name of the bottle to install into has
        been modified.
        FIXME: Specify if one can receive such notifications when
        installing in a new bottle.
        """

    @cxobjc.delegate
    def profileMediaAdded_(self, filename):
        """Notifies the GUI that a non-mountpoint source was found for the
        selected profile and added to profile_medias.
        """

    @cxobjc.delegate
    def profileMediaRemoved_(self, filename):
        """Notifies the GUI that a non-mountpoint source in profile_medias was
        removed because it no longer applies to the selected profile.
        """

_NULL_DELEGATE = InstallTaskDelegate()

def _do_nothing(*_args, **_kwargs):
    pass


#####
#
# InstallTask
#
#####

class InstallTask(cxobjc.Proxy):
    """The InstallTask class will keep track of the state of a user-selected
    c4 profile as we go through the process of installing it. It will also
    provide an interface to installation information that must be returned at
    run-time, e.g. bottle compatibility.
    """

    # FIXME: Rename this to something more descriptive

    def __init__(self, delegate=None):
        cxobjc.Proxy.__init__(self)
        self._profiles = None

        # An object with methods that will be called when attributes change.
        self._delegate = _NULL_DELEGATE
        self.set_delegate(delegate)
        self._queued_delegate_calls = []

        # The C4 profile of the application to install
        self._profile = None
        self._profile_requested = False

        self.installerLocale = None

        # Determine the locale to use for profiles without a language selection
        for langid in cxutils.get_preferred_languages():
            if langid != '' and langid in c4profiles.LANGUAGES:
                self._userInstallerLocale = langid
                break
        else:
            self._userInstallerLocale = ''

        # The locale we will actually use for installs. This is equal to
        # installerLocale if the current profile has a language selection,
        # otherwise it's equal to _userInstallerLocale
        self.realInstallerLocale = None

        # The corresponding installer profile. Note that it may belong to
        # another C4 profile, such as the profile for unknown applications.
        self.installer = None

        # A mapping of absolute paths to the corresponding SourceMedia
        # objects. Only the GUI knows about and keeps track of mount points,
        # etc. So it is responsible for filling in this map and keeping it up
        # to date.
        self.source_medias = {}

        # A set of automatically-detected sources, based on the current profile.
        self.profile_medias = set()

        # The source media.
        self._installer_source_requested = False # True if the user requested this source
        self.installerDownloadSource = None
        self.installerSource = None

        self.installWithSteam = False

        # Whether this install task should apply cxfixes or not
        self.apply_cxfixes = True

        # A set of profiles to include or exclude from the install, overriding profiles.
        self.dependency_overrides = {}

        # The target bottle
        self.bottles = {}
        self.templates = {}
        for template in bottlemanagement.template_list():
            self.templates[template] = TargetTemplate(self, template)
        self.targetBottle = None
        self._target_bottle_requested = False
        self.newBottleName = ""
        self._new_bottle_name_requested = False

    @cxobjc.python_method
    def _getprofiles(self):
        if self._profiles is None:
            self._profiles = c4profilesmanager.C4ProfilesSet.all_profiles()

        return self._profiles

    @cxobjc.python_method
    def _setprofiles(self, value):
        self._profiles = value
        if self._profile:
            if self._profile.appid in self.profiles:
                self._setprofile(self.profiles[self._profile.appid], requested=self._profile_requested)
            else:
                self.profile = None

    profiles = property(_getprofiles, _setprofiles)


    # Special initializer for objc. This must /always/ be called explicitly
    # on the Mac.
    def initWithDelegate_(self, delegate):
        self = cxobjc.Proxy.nsobject_init(self)
        if self is not None:
            self.__init__(delegate)
        return self

    def init(self):
        return self.initWithDelegate_(None)

    @cxobjc.namedSelector(b'setDelegate:')
    def set_delegate(self, delegate):
        if delegate:
            self._delegate = delegate
        else:
            self._delegate = _NULL_DELEGATE

    @cxobjc.python_method
    def _queue_delegate_call(self, methodname, arg1=None):
        self._queued_delegate_calls.append((methodname, (arg1,)))

    @cxobjc.python_method
    def _flush_delegate_calls(self):
        calls = self._queued_delegate_calls
        self._queued_delegate_calls = []
        for methodname, args in calls:
            args = args[0:InstallTaskDelegate.__dict__[methodname].__code__.co_argcount-1]
            getattr(self._delegate, methodname, _do_nothing)(*args)

    @cxobjc.python_method
    def select_profile_from_source(self):
        if self.installerSource and self._installer_source_requested and not self._profile_requested:
            # Try to select a profile based on the source media.
            if self.installerSource in self.source_medias:
                profiles = self.source_medias[self.installerSource].profile_ids
                if len(profiles) == 1:
                    self._setprofile(next(iter(profiles.values())), requested=False)

    @cxobjc.namedSelector(b'autoFillSettings')
    def auto_fill_settings(self):
        # After all initial settings have been set, and this object and the
        # change delegate are initialized, try to guess at anything remaining.

        if not self._profile_requested and not self._installer_source_requested:
            # If we can find exactly one profile to select based on media, select it.
            package_to_select = None
            for source in self.source_medias.values():
                profiles = source.profile_ids
                if len(profiles) == 1:
                    if package_to_select:
                        # too many profiles
                        break
                    package_to_select = next(iter(profiles.values()))
                elif profiles:
                    # too many profiles
                    break
            else:
                # 0 or 1 profiles found
                if package_to_select:
                    self._setprofile(package_to_select, requested=False)

        self.select_profile_from_source()

    @cxobjc.namedSelector(b'volumeMatchesProfile:')
    def volume_matches_profile(self, volume):
        return self.profile and volume in self.source_medias and \
            self.profile.appid in self.source_medias[volume].profile_ids

    #####
    #
    # Profile selection
    #
    #####

    @cxobjc.python_method
    def _get_condition_languages(self, condition, result=None):
        if result is None:
            result = set()
        if isinstance(condition, c4profiles.C4ConditionUnary):
            self._get_condition_languages(condition.child, result)
        elif isinstance(condition, c4profiles.C4ConditionNary):
            for i in condition.children:
                self._get_condition_languages(i, result)
        elif isinstance(condition, (c4profiles.C4ConditionCompare, c4profiles.C4ConditionMatch)):
            if condition.name == 'locale':
                result.add(condition.value)
        return result

    @staticmethod
    @cxobjc.python_method
    def _scan_url_locales(urls, default_locales, nondefault_locales):
        # urls may be a dictionary of strings to string, or strings to tuple, that doesn't matter for our usage.
        if '' in urls:
            default_url = urls['']
        else:
            default_url = None

        for locale, url in urls.items():
            if url == default_url:
                default_locales.add(locale)
            else:
                nondefault_locales.add(locale)

    @cxobjc.python_method
    def profile_languages(self, profile=None):
        "Returns the set of languages that should be presented to the user for a profile"
        if profile is None:
            profile = self._profile

        if profile is not None:
            use_if_languages = set()
            for installer_profile in profile.installer_profiles:
                use_if_languages.update(self._get_condition_languages(installer_profile.use_if))

            # Get a list of the locales that are equivalent to the default,
            # and of the others, for both the download pages and urls
            default_locales = set()
            nondefault_locales = set()

            if profile.app_profile.download_urls:
                self._scan_url_locales(profile.app_profile.download_urls, default_locales, nondefault_locales)
                self._scan_url_locales(profile.app_profile.download_urls_64bit, default_locales, nondefault_locales)
            else:
                self._scan_url_locales(profile.app_profile.download_page_urls, default_locales, nondefault_locales)

            result = set()

            if use_if_languages:
                result.update(use_if_languages)
                # If we have programmed special behavior for a particular set of
                # languages, selecting a language not in the set must be
                # possible.
                result.add('')

            if result or nondefault_locales:
                result.update(nondefault_locales)
                result.update(default_locales)

            # FIXME: Account for dependencies?

            return result

        return ()

    # Wrapper for Objective-C
    def profileLanguages(self):
        return list(self.profile_languages())

    @cxobjc.python_method
    def _getprofile(self):
        return self._profile

    @cxobjc.python_method
    def _setprofile(self, profile, requested=True):
        if self._profile == profile:
            return

        if profile is None:
            requested = False

        if profile is not None and profile.appid in self.dependency_overrides:
            del self.dependency_overrides[profile.appid]

        hadProfile = self._profile is not None

        # Profile
        self._profile = profile
        self._profile_requested = requested

        # Locale
        if profile:
            languages = self.profile_languages(profile)
            if languages:
                # Try to find an installer matching one of the user's preferred
                # locales
                for langid in cxutils.get_preferred_languages():
                    if langid != '' and langid in languages:
                        self.installerLocale = langid
                        break
                else:
                    # Barring that, pick the one locale that the application is
                    # available for.
                    if len(languages) == 1:
                        self.installerLocale = next(iter(languages))
                    else:
                        # But if there are multiple options we have no reason to
                        # pick one over the other, so select 'Other Language'
                        # and let the user decide.
                        self.installerLocale = ''
                self.realInstallerLocale = self.installerLocale
            else:
                self.installerLocale = ''
                self.realInstallerLocale = self._userInstallerLocale

        # Source Media

        # If the user had requested a download, we must discard the user's request.
        if self.installerDownloadSource is not None or self.installWithSteam:
            self._clear_installer_source()

        old_profile_medias = self.profile_medias
        if profile:
            self.profile_medias = set(downloaddetector.find_installers(
                self.download_search_paths(), c4profilesmanager.C4ProfilesSet.from_profile(profile)))
        else:
            self.profile_medias = set()

        for media in old_profile_medias - self.profile_medias:
            self._queue_delegate_call('profileMediaRemoved_', media)
        for media in self.profile_medias - old_profile_medias:
            self._queue_delegate_call('profileMediaAdded_', media)

        if not self._installer_source_requested: # pylint: disable=R1702
            if profile:
                installerSource = None
                installerDownloadSource = None
                multipleSources = False

                builtin = cxaiemedia.get_builtin_installer(self._profile.appid)
                if builtin is not None:
                    installerSource = builtin
                else:
                    for source in self.source_medias.values():
                        if self._profile.appid in source.profile_ids:
                            if installerSource is None:
                                installerSource = source.path
                            else:
                                # If there are multiple matches, then let the user
                                # pick one
                                installerSource = None
                                multipleSources = True
                                break

                    if not installerSource:
                        installerDownloadSource = self._get_download_url()

                if installerSource:
                    self.set_installer_source(installerSource, requested=False, flush_delegate=False)
                elif installerDownloadSource and not self.steamid:
                    self.set_installer_download(requested=False, flush_delegate=False)
                elif len(self.profile_medias) == 1:
                    self.set_installer_source(list(self.profile_medias)[0], requested=False, flush_delegate=False)
                elif self.steamid and not installerDownloadSource:
                    self.set_installer_steam(requested=False, flush_delegate=False)
                elif multipleSources or hadProfile:
                    self._clear_installer_source()

            else:
                self._clear_installer_source()

        self._queue_delegate_call('profileChanged')

        self._trigger_bottle_analysis()

        self._flush_delegate_calls()

    profile = property(_getprofile, _setprofile)

    # Wrapper for Objective-C
    def setProfile_(self, profile):
        self.profile = profile

    @cxobjc.namedSelector(b'updateProfiles')
    def update_profiles(self):
        self._profiles = None
        if self._profile:
            if self._profile.appid in self.profiles:
                self._setprofile(self.profiles[self._profile.appid], requested=self._profile_requested)
            else:
                self.profile = None


    @cxobjc.namedSelector(b'useAutorunFile:')
    def use_autorun_file(self, c4pfile):
        if self._profile_requested:
            return
        autorun = c4pfile.get_autorun_id()
        if autorun and autorun in self.profiles:
            self._setprofile(self.profiles[autorun], True)


    @cxobjc.python_method
    def _is_virtual(self):
        return self._profile and 'virtual' in self._profile.app_profile.flags

    virtual = property(_is_virtual)


    @cxobjc.python_method
    def _get_display_name(self):
        if self._profile:
            return self._profile.name
        return None

    displayName = property(_get_display_name)


    @cxobjc.namedSelector(b'setLocale:')
    def SetLocale(self, locale):
        if self.installerLocale == locale:
            return

        self.realInstallerLocale = self.installerLocale = locale

        # Recalculate the download url if applicable
        if self.installerDownloadSource or not self.installerSource:
            self.set_installer_download(requested=False, flush_delegate=False)

        if self._profile:
            # A change of locale may impact the bottle selection
            self._trigger_bottle_analysis()

        self._flush_delegate_calls()


    @cxobjc.namedSelector(b'addDependencyOverrides:ofType:')
    def AddDependencyOverrides(self, appids, override_type=OVERRIDE_INCLUDE):
        for appid in appids:
            self.dependency_overrides[appid] = override_type

        self._trigger_bottle_analysis()

        self._flush_delegate_calls()

    @cxobjc.namedSelector(b'removeDependencyOverrides:')
    def RemoveDependencyOverrides(self, appids):
        for appid in appids:
            if appid in self.dependency_overrides:
                del self.dependency_overrides[appid]

        self._trigger_bottle_analysis()

        self._flush_delegate_calls()

    def ClearDependencyOverrides(self):
        self.dependency_overrides = {}

        self._trigger_bottle_analysis()

        self._flush_delegate_calls()

    _component_categories = set(['Component',
                                 'Component/Font',
                                 'Component/Fonts',
                                 'Non Applications/CrossTie Snippets',
                                 'Non-Applications/CrossTie Snippets',
                                 'Non-Applications/Components',
                                 'Non-Applications/Components/Fonts'])

    def SuggestedDependencyOverrides(self):
        main_appid = self.profile.appid if self.profile else None
        if self.target_bottle and self.target_bottle.bottle and \
           self.target_bottle.bottle.installed_packages_ready:
            installed_packages = self.target_bottle.bottle.installed_packages
        else:
            installed_packages = ()
        result = []
        for appid, profile in self.profiles.items():
            if profile is self._profile or appid in installed_packages or \
               not (profile.app_profile.download_urls or 'virtual' in profile.app_profile.flags):
                continue

            if main_appid in profile.app_profile.extra_fors or \
               profile.app_profile.raw_category in self._component_categories:
                result.append(appid)

        return result


    #####
    #
    # Source media selection
    #
    #####

    @cxobjc.namedSelector(b'addSourceMedia:withLabel:andDevice:')
    def add_source_media(self, mountpoint, label='', device=''):
        self.source_medias[mountpoint] = SourceMedia(mountpoint, label, device)

    @cxobjc.namedSelector(b'removeSourceMedia:')
    def remove_source_media(self, mountpoint):
        if mountpoint in self.source_medias:
            del self.source_medias[mountpoint]

        if self.installerSource == mountpoint:
            self._clear_installer_source(flush_delegate=True)

    # On macOS, don't search user directories since it triggers annoying permission dialogs
    @staticmethod
    @cxobjc.python_method
    def download_search_paths():
        if distversion.IS_MACOSX:
            return ()
        return (os.environ.get('HOME', '/'), cxutils.get_download_dir(), cxutils.get_desktop_dir())

    @cxobjc.python_method
    def set_installer_source(self, inSource, requested=True, flush_delegate=True):
        if self.installerSource == inSource:
            return

        self.installerSource = inSource

        if inSource:
            if requested:
                self.add_source_media(inSource)

            self.installerDownloadSource = None
            self.installWithSteam = False
            self._installer_source_requested = requested
            self.select_profile_from_source()

        self._queue_delegate_call('sourceChanged')

        self._trigger_bottle_analysis()

        if flush_delegate:
            self._flush_delegate_calls()

    # Wrapper for Objective-C
    def setInstallerSource_Requested_(self, inSource, inRequested):
        self.set_installer_source(inSource, inRequested)

    @cxobjc.python_method
    def set_installer_download(self, requested=True, flush_delegate=True):
        url = self._get_download_url()
        if self.installerDownloadSource == url:
            return

        if url is None:
            self._clear_installer_source()
            return

        self.installerSource = None
        self.installerDownloadSource = url
        self.installWithSteam = False
        self._installer_source_requested = requested

        self._queue_delegate_call('sourceChanged')

        self._trigger_bottle_analysis()

        if flush_delegate:
            self._flush_delegate_calls()

    # Wrapper for Objective-C
    def setInstallerDownload(self):
        self.set_installer_download()

    @cxobjc.python_method
    def set_installer_steam(self, requested=True, flush_delegate=True):
        source = "steam://install/%s" % self._profile.app_profile.steamid
        if self.installerSource == source:
            return

        self.installerSource = source
        self._installer_source_requested = requested
        self.installWithSteam = True
        self.installerDownloadSource = None

        self._queue_delegate_call('sourceChanged')

        self._trigger_bottle_analysis()

        if flush_delegate:
            self._flush_delegate_calls()

    # Wrapper for Objective-C
    def setInstallerSteam(self):
        self.set_installer_steam()

    @cxobjc.python_method
    def _get_download_url(self):
        # Returns the download URL of the currently selected locale
        if not self._profile:
            return None

        if self._profile.app_profile.download_urls:
            if self._profile.app_profile.download_urls_64bit and self.is64bit():
                download_urls = self._profile.app_profile.download_urls_64bit
            else:
                download_urls = self._profile.app_profile.download_urls

            if self.installerLocale in download_urls:
                return download_urls[self.installerLocale][0]

            if '' in download_urls:
                return download_urls[''][0]

        return None

    download_url = property(_get_download_url)

    @cxobjc.python_method
    def _get_steamid(self):
        if not self._profile:
            return None

        if not self._profile.app_profile:
            return None

        return self._profile.app_profile.steamid

    steamid = property(_get_steamid)

    @cxobjc.python_method
    def _clear_installer_source(self, flush_delegate=False):
        if self.installerSource is None and self.installerDownloadSource is None:
            return

        self.installerSource = None
        self.installerDownloadSource = None
        self._installer_source_requested = False
        self.installWithSteam = False

        self._queue_delegate_call('sourceChanged')

        self._trigger_bottle_analysis()

        if flush_delegate:
            self._flush_delegate_calls()

    def _has_installer_source(self):
        return self.installerSource or self.installerDownloadSource or self.virtual or self.installWithSteam

    has_installer_source = property(_has_installer_source)


    #####
    #
    # Automatic bottle selection
    #
    #####

    def _take_bottle_snapshot(self):
        self._bottle_snapshot = {
            'targetBottle': self.targetBottle,
            'newBottleName': self.newBottleName,
            }

    def _send_bottle_change_notifications(self):
        """An internal helper for sending the relevant notification(s) when
        the selected bottle gets changed.
        """
        if not self._bottle_snapshot:
            cxlog.err('No bottle snapshot was taken!')
            return
        if self.targetBottle != self._bottle_snapshot['targetBottle']:
            self._queue_delegate_call('bottleCreateChanged')
            self._queue_delegate_call('bottleTemplateChanged')
            self._queue_delegate_call('bottleNameChanged')
        if self.newBottleName != self._bottle_snapshot['newBottleName']:
            self._queue_delegate_call('bottleNewnameChanged')

    def _unpick_bottle(self):
        """This gets called when the target profile is unset and resets the
        relevant fields.
        """
        self._take_bottle_snapshot()
        if not self._target_bottle_requested:
            self.targetBottle = None
        if not self._new_bottle_name_requested:
            self.newBottleName = None
        self._send_bottle_change_notifications()

    def _pick_new_bottle_name(self):
        """Picks a suitable name for the new bottle if we are free to do so.

        The caller is responsible for sending the bottle change notification(s).
        """
        if not self._new_bottle_name_requested:
            # Note that here we know that the new bottle name is necessarily
            # valid.
            bottle_name_hint = self._profile.name
            if self._profile.is_unknown and self.installerSource:
                if self.installerSource in self.source_medias and \
                   self.source_medias[self.installerSource].label:
                    bottle_name_hint = self.source_medias[self.installerSource].label
                else:
                    bottle_name_hint = os.path.basename(self.installerSource.rstrip('/')).split('.', 1)[0]

            self.newBottleName = bottlequery.unique_bottle_name(cxutils.sanitize_bottlename(bottle_name_hint))

    def _get_new_bottle_template(self):
        """Returns a suitable template for the new bottle."""

        # Scan the profile's preferred template list and pick the first one
        # which is actually compatible
        fallback = None
        for template in self._profile.app_profile.bottle_types['use']:
            if template not in self.templates:
                # Not supported in the CrossOver version!
                pass
            elif self.templates[template].get_category() == CAT_COMPATIBLE:
                return template
            elif not fallback:
                fallback = template

        if fallback:
            # No template is compatible with both the application and all its
            # dependencies. So ignore the dependencies.
            return fallback

        # None of this application's templates is supported by this CrossOver
        # version! So pick the same one as for the unknown profile.
        return self.profiles.unknown_profile().app_profile.preferred_bottle_type

    new_bottle_template = property(_get_new_bottle_template)

    def _pick_new_bottle(self):
        """Picks a suitable template for the new bottle if we are free to do so.

        The caller is responsible for sending the bottle change notification(s).
        """
        if self._target_bottle_requested:
            return

        self.targetBottle = self.templates[self.new_bottle_template]

    def _pick_bottle(self):
        """This method performs the automatic bottle selection when
        appropriate.

        It is a helper for _categorize_bottles() and must only be called when
        all bottles have been assigned a category.
        """
        self._take_bottle_snapshot()
        if not self._target_bottle_requested:
            self.targetBottle = None
            for target_bottle in self.bottles.values():
                if target_bottle.get_category() != CAT_RECOMMENDED:
                    pass
                elif self.targetBottle:
                    # We have more than one recommended bottle so we need
                    # the user to pick one
                    self.targetBottle = None
                    break
                else:
                    self.targetBottle = target_bottle
            if not self.targetBottle and ('component' not in self._profile.app_profile.flags or 'application' in self._profile.app_profile.flags):
                self._pick_new_bottle()
        self._pick_new_bottle_name()
        self._send_bottle_change_notifications()


    #####
    #
    # Bottle analysis and categorization
    #
    #####

    @cxobjc.python_method
    def _trigger_bottle_analysis(self):
        """Triggers a bottle analysis to:
        - determine which ones are compatible or incompatible with the
          selected profile and installation source,
        - determine which ones to recommend installation into (if any),
        - pick a bottle if possible,
        - and acquire all the data we need to start the installation.
        """
        # Should this prove too slow this will remain the entry point for
        # triggering the bottle analysis but part of the work will be done in
        # the background.
        if self._profile:
            self._categorize_bottles()
        else:
            # Note that we don't need to go through all the target bottle
            # objects to reset their categories thanks to their cache mechanism.
            # So all we need is to unset the selected bottle if appropriate.
            self._unpick_bottle()
            # Notify the GUI that the category of each bottle and template
            # has changed.
            for target in self.templates.values():
                self._queue_delegate_call('categorizedBottle_', target)
            for target in self.bottles.values():
                self._queue_delegate_call('categorizedBottle_', target)
            # Also notify it that we have recategorized all the bottles
            self._queue_delegate_call('categorizedAllBottles')

    @cxobjc.python_method
    def _categorize_list(self, target_iter):
        """This is a helper for _categorize_bottles()."""
        categorized_all = True
        for target in target_iter:
            if not target.analyzed() and target.analyze():
                # Notify the GUI that we have freshly analyzed this
                # target bottle
                self._queue_delegate_call('analyzedBottle_', target)
            if not target.has_category():
                if target.get_category() != CAT_NONE:
                    # Notify the GUI that we have freshly categorized this
                    # target bottle
                    self._queue_delegate_call('categorizedBottle_', target)
                else:
                    categorized_all = False
        return categorized_all

    @cxobjc.python_method
    def _categorize_bottles(self):
        """This goes through all the bottles and templates trying to determine
        their category. If it was successful it then calls _pick_bottle() and
        notifies the GUI.
        """
        if self._categorize_list(self.templates.values()) and \
           self._categorize_list(self.bottles.values()):
            # We now know the category of all the bottles and templates and
            # thus we can now pick one
            self._pick_bottle()
            # The GUI may have been waiting for all the categories to be known
            # to display the bottle list. So notify it we're done.
            self._queue_delegate_call('categorizedAllBottles')
        # else:
        #     This means we are waiting for one (or more) of the bottles
        #     installed applications list to compute the categories. So this
        #     method will be called again when installed_applications_ready()
        #     gets called for the relevant bottles.


    #####
    #
    # Bottle addition and removal
    #
    #####

    @cxobjc.namedSelector(b'addBottle:')
    def add_bottle(self, bottle):
        """Notifies InstallTask that a new bottle was detected.

        The caller MUST then schedule the detection of that bottle's installed
        applications and, when that's done, it MUST notify InstallTask through
        installed_applications_ready(). The one exception is if the bottle
        disappears before then and in that case it MUST notify InstallTask
        through remove_bottle().

        This MUST be called in the main thread.
        """
        self.bottles[bottle.name] = TargetBottle(self, bottle)
        return self.bottles[bottle.name]

    @cxobjc.namedSelector(b'installedApplicationsReady:')
    def installed_applications_ready(self, bottle):
        """Notifies InstallTask that the specified bottle's installed
        applications list is now available.

        This MUST be called in the main thread.
        """
        # FIXME: Scan the installed applications list and update the extrafors.
        target_bottle = self.bottles.get(bottle.name)
        if not target_bottle or target_bottle.installed_packages_ready:
            return

        if bottle.installed_packages_ready:
            target_bottle.installed_packages_ready = True

            self.trigger_bottle_analysis()

    @cxobjc.namedSelector(b'triggerBottleAnalysis')
    def trigger_bottle_analysis(self):
        """Notifies InstallTask that the bottles may
        have changed while notifying installed_applications_ready
        was disabled.

        This MUST be called in the main thread.
        """
        self._trigger_bottle_analysis()
        self._flush_delegate_calls()

    @cxobjc.python_method
    def remove_bottle_by_name(self, bottlename):
        """Notifies InstallTask that the specified bottle has been removed.

        This may be called _instead of_ installed_applications_ready() if the
        removal happened before the installed application detection could
        complete. In this case installed_applications_ready() must not be
        called for that bottle.

        It is also a bug to call remove_bottle_by_name() twice or to call it on
        unregistered bottles.

        This MUST be called in the main thread.
        """
        del self.bottles[bottlename]
        # FIXME: Remove this bottle from the extrafors
        if self.targetBottle and self.targetBottle.bottlename == bottlename:
            self._take_bottle_snapshot()
            self.targetBottle = None
            self._send_bottle_change_notifications()

        # The bottle removal does not impact any of the bottle categories but
        # it may impact the automatic bottle selection. So review things again.
        self.trigger_bottle_analysis()

    @cxobjc.namedSelector(b'removeBottle:')
    def remove_bottle(self, bottle):
        """Use remove_bottle_by_name instead."""
        self.remove_bottle_by_name(bottle.name)

    #####
    #
    # Target bottle selection
    #
    #####

    @cxobjc.namedSelector(b'isNewBottleNameValid:')
    def is_new_bottle_name_valid(self, new_bottle_name): # pylint: disable=R0201
        """Returns True if the new bottle name is valid and False otherwise.

        In particular this checks for that it does not match an existing
        bottle name, file or directory.
        This MUST be called in the main thread.
        """
        return bottlequery.is_valid_new_bottle_name(new_bottle_name)

    @cxobjc.python_method
    def _set_target_bottle(self, target_bottle):
        requested = True
        if isinstance(target_bottle, TargetTemplate):
            if self.templates[target_bottle.template] != target_bottle:
                raise ValueError()
        elif target_bottle is not None:
            if self.bottles[target_bottle.bottlename] != target_bottle:
                raise ValueError()
        else:
            requested = False

        if target_bottle != self.targetBottle:
            was_64bit = self.is64bit()

            self.targetBottle = target_bottle
            self._target_bottle_requested = requested

            # Recalculate the download url if applicable
            if self.installerDownloadSource and self.is64bit() != was_64bit:
                self.set_installer_download(requested=False, flush_delegate=False)

            self._queue_delegate_call('bottleCreateChanged')
            self._queue_delegate_call('bottleTemplateChanged')
            self._queue_delegate_call('bottleNameChanged')

            self._flush_delegate_calls()

    @cxobjc.python_method
    def _get_target_bottle(self):
        """Returns the TargetBottle object for the bottle selected for
        installation. This object can then be used to get all the details about
        that bottle: its category, the dependencies to install, and potential
        issues.

        Note also that this works equally well for existing bottles and new
        bottles.
        """
        return self.targetBottle

    target_bottle = property(_get_target_bottle, _set_target_bottle)

    def _get_new_bottle_name(self):
        return self.newBottleName

    @cxobjc.namedSelector(b'setNewBottleName:')
    def _set_new_bottle_name(self, new_bottle_name):
        if self.newBottleName != new_bottle_name:
            self.newBottleName = new_bottle_name
            self._new_bottle_name_requested = True

            self._queue_delegate_call('bottleNewnameChanged')
            self._flush_delegate_calls()

    new_bottle_name = property(_get_new_bottle_name, _set_new_bottle_name)

    @cxobjc.python_method
    def set_create_new_bottle(self, newbottlename, template):
        if newbottlename is None:
            newbottlename = self.new_bottle_name

        if template is None:
            template = self.target_template

        self.new_bottle_name = newbottlename
        self.target_bottle = self.templates[template]

    # Wrapper for Objective-C
    def createNewBottle_template_(self, newbottlename, template):
        self.set_create_new_bottle(newbottlename, template)

    def GetCreateNewBottle(self):
        return isinstance(self.targetBottle, TargetTemplate)

    create_new_bottle = property(GetCreateNewBottle)

    @cxobjc.python_method
    def _get_existing_bottle_name(self):
        if self.targetBottle:
            return self.targetBottle.bottlename
        return None

    @cxobjc.python_method
    def _get_bottle_name(self):
        if isinstance(self.targetBottle, TargetTemplate):
            return self.newBottleName
        if self.targetBottle:
            return self.targetBottle.bottlename
        return None

    @cxobjc.python_method
    def _set_bottle_name(self, name):
        if name:
            self.target_bottle = self.bottles[name]
        else:
            self.target_bottle = None

    bottlename = property(_get_bottle_name, _set_bottle_name)

    def _gettarget_template(self):
        if self.targetBottle:
            return self.targetBottle.template
        return None

    target_template = property(_gettarget_template)

    def is64bit(self):
        template = self.target_template
        if template is None:
            return False
        return template.endswith('_64')

    #####
    #
    # Getting the summary data
    #
    #####

    def get_installer_profile(self):
        """Computes the appropriate installer profile for the selected
        profile, locale and bottle type combination."""
        # FIXME: This method should go away and the code that uses it should
        # use or be migrated to target_bottle instead.
        target_bottle = self.targetBottle
        if target_bottle is None:
            if self._profile.app_profile.preferred_bottle_type:
                target_bottle = self.templates[self._profile.app_profile.preferred_bottle_type]
            else:
                target_bottle = self.templates[self.profiles.unknown_profile().app_profile.preferred_bottle_type]
        elif not target_bottle.analyzed():
            target_bottle = self.templates[target_bottle.template]
        return target_bottle.installprofile.copy()

    @cxobjc.python_method
    def get_profile_names(self, appids):
        """Converts a list of profile ids into a list of human-readable
        profile names."""
        installers = self.target_bottle.installers
        profile_names = []
        for appid in appids:
            profile_names.append(installers[appid].parent.name)
        return profile_names

    def _get_installation_notes(self):
        return self.get_installer_profile().installation_notes

    installationNotes = property(_get_installation_notes)


    def _get_post_install_url(self):
        return self.get_installer_profile().post_install_url

    post_install_url = property(_get_post_install_url)

    _RE_SINGLE = re.compile('^[a-zA-Z0-9:_-]*$', re.IGNORECASE)

    @cxobjc.python_method
    def _find_warning_in_mapping(self, warning, mapping):
        # Search for the given key in a cxdiag mapping
        if not self._RE_SINGLE.match(warning):
            # A regular expression
            regex = re.compile(warning, re.IGNORECASE)
            for key in mapping:
                if regex.match(key):
                    yield key
        elif warning in mapping:
            yield warning

    @cxobjc.python_method
    def _filter_cxdiag_messages(self, messages, profile):
        # Only report issues that are listed in the profile except for the
        # 'required' ones because they are critical for all applications, and
        # the 'recommended' ones because we consider those to be strongly
        # recommended too.
        filtered = {}
        for key, issue in messages.items():
            if issue[0] == 'required' or issue[0] == 'recommended':
                filtered[key] = issue

        # Filter out the remaining issues and adjust their level according to
        # the profile specifications.
        for check in profile.cxdiag_checks:
            check = check.lower()
            if check.startswith('apprequire:') or check.startswith('apprecommend:'):
                issue_id = check.split(':', 1)[1]
                new_level = 'suggested'
                if check.startswith('apprequire:'):
                    new_level = 'required'
                elif check.startswith('apprecommend:'):
                    new_level = 'recommended'
                for key in self._find_warning_in_mapping(issue_id, messages):
                    filtered[key] = messages[key]._replace(level=new_level)
            elif check.startswith('ignore:'):
                issue_id = check.split(':', 1)[1]
                ignore_list = list(self._find_warning_in_mapping(issue_id, messages))
                for key in ignore_list:
                    if key in filtered:
                        del filtered[key]
            elif check == 'closed':
                # We now consider all lists to be closed so this is a no-op
                break
            else:
                cxlog.warn("unrecognized cxdiagcheck string: %s" % cxlog.to_str(check))

        return filtered

    @staticmethod
    @cxobjc.python_method
    def _cxdiag_level_cmp(a, b):
        levels = ['suggested', 'recommended', 'required']
        try:
            return cxutils.cmp(levels.index(a), levels.index(b))
        except ValueError:
            return 0

    @cxobjc.python_method
    def _application_name(self):
        if not self._profile:
            return None

        if self._profile.is_unknown:
            return None

        return self.displayName

    application_name = property(_application_name)

    @cxobjc.python_method
    def get_cxdiag_messages(self):
        """Returns a dictionary of cxdiag messages that apply to the current
        profile."""

        if cxproduct.is_wow64_install():
            flags = cxdiag.CHECK_64BIT
        else:
            flags = cxdiag.CHECK_32BIT
            if self.is64bit():
                flags |= cxdiag.CHECK_64BIT
        diag = cxdiag.get(self._get_existing_bottle_name(), flags)

        # Add the masked issues to the list
        cxfixes.clear_errors()
        for errid, title in diag.warnings.items():
            cxfixes.add_error(errid, title)
        cxfixes.add_masked_errors(set(('required', 'recommended', 'suggested')))
        allwarnings = cxfixes.get_errors()
        cxfixes.clear_errors()

        if self._profile is not None and self.target_bottle and self.target_bottle.analyzed():
            result = {}
            for profile in self.target_bottle.installers.values():
                profile_messages = self._filter_cxdiag_messages(allwarnings, profile)
                for key, message in profile_messages.items():
                    if key not in result or \
                       self._cxdiag_level_cmp(message[0], result[key][0]) > 0:
                        result[key] = message
        else:
            result = allwarnings.copy()

        return result

    @cxobjc.python_method
    def get_apply_cxfixes(self):
        """True if this task should apply necessary cxfixes, False otherwise."""
        return self.apply_cxfixes

    @cxobjc.python_method
    def get_summary_string(self):
        lines = []
        if self.profile:
            lines.append('Installing: %s\n' % self.profile.name)
        if self.installerLocale:
            lines.append('Locale: %s\n' % self.installerLocale)
        if self.bottlename:
            lines.append('Bottle: %s\n' % self.bottlename)
        if self.installWithSteam:
            lines.append('From Steam: %s\n' % self.installerSource)
        elif self.installerSource:
            lines.append('From file: %s\n' % self.installerSource)
        elif self.installerDownloadSource:
            lines.append('From download url: %s\n' % self.installerDownloadSource)
        for k, v in self.dependency_overrides.items():
            if k in self.profiles:
                if v == OVERRIDE_EXCLUDE:
                    lines.append('Manually removed from install: %s\n' % self.profiles[k].name)
                elif v == OVERRIDE_INCLUDE:
                    lines.append('Manually added to install: %s\n' % self.profiles[k].name)
        return ''.join(lines)

    @cxobjc.namedSelector(b'getDependencyDetails')
    def get_dependency_details(self):
        result = {'errors': [], 'warnings': [], 'notes': []}

        has_dependencies = False
        if not distversion.IS_MACOSX:
            for errid in self.get_cxdiag_messages():
                packages = cxfixes.get_packages(errid)
                if packages and self.apply_cxfixes:
                    has_dependencies = True

        if self.target_bottle and self.target_bottle.analyzed() and \
           len(self.target_bottle.installers) >= 2:
            has_dependencies = True

        if self.virtual:
            if has_dependencies:
                result['notes'].append(_("CrossOver will install dependencies only"))
                has_dependencies = False
            else:
                result['notes'].append(_("CrossOver will make configuration changes only"))

        if has_dependencies:
            result['notes'].append(_("CrossOver will also install additional dependencies"))

        if self.target_bottle and self.target_bottle.analyzed():
            include_names = []
            exclude_names = []
            for appid, override in self.dependency_overrides.items():
                if appid not in self.target_bottle.reasons:
                    continue

                if appid not in self.profiles:
                    continue

                if override == OVERRIDE_INCLUDE:
                    include_names.append(self.profiles[appid].name)

                if override == OVERRIDE_EXCLUDE:
                    exclude_names.append(self.profiles[appid].name)

            if include_names:
                result['notes'].append(_("CrossOver will also install the following manually added items: %s") % cxutils.html_escape(', '.join(include_names)))

            if exclude_names:
                result['warnings'].append(_("The following dependencies were manually excluded from the install: %s") % cxutils.html_escape(', '.join(exclude_names)))

        return result

    @cxobjc.namedSelector(b'getInstallerDetails')
    def get_installer_details(self):
        result = {'errors': [], 'warnings': [], 'notes': []}

        if self.virtual:
            pass
        elif self.installWithSteam:
            if self.application_name:
                result['notes'].append(_("CrossOver will install '%(application)s' via Steam") % {
                    'application': self.application_name})
            else:
                result['notes'].append(_("CrossOver will install this software via Steam"))
        elif self.installerSource:
            if self.application_name:
                result['notes'].append(_("CrossOver will install '%(application)s' from %(location)s") % {
                    'application': self.application_name, 'location': cxutils.html_escape(self.installerSource)})
            else:
                result['notes'].append(_("CrossOver will install this software from %(location)s") % {
                    'location': cxutils.html_escape(self.installerSource)})
        elif self.installerDownloadSource:
            result['notes'].append(_("CrossOver will download the installer from %s") % cxutils.html_escape(self.installerDownloadSource))
        else:
            if self.application_name:
                label = _("You need to provide the installer file or disk in order to install '%(application)s' using CrossOver") % {
                    'application': self.application_name}
            else:
                label = _("You need to provide the installer file or disk in order to install this software using CrossOver")
            if self.profile and self.profile.app_profile.download_page_url:
                label = _("You need to provide the installer file or disk in order to install this software using CrossOver. If you don't have an installer, it can be downloaded from %s") % ('<a href="%s">%s</a>' % (self.profile.app_profile.download_page_url, cxutils.html_escape(self.profile.app_profile.download_page_url)))

            result['warnings'].append(label)

        return result

    @cxobjc.namedSelector(b'getBottleDetails')
    def get_bottle_details(self):
        result = {'errors': [], 'warnings': [], 'notes': []}

        if not self.bottlename:
            if self.application_name:
                result['warnings'].append(_("You need to select the bottle to install '%(application)s' into") % {
                    'application': self.application_name})
            else:
                result['warnings'].append(_("You need to select the bottle to install this software into"))
        elif self.target_bottle and self.target_bottle.analyzed():
            if self.create_new_bottle and self.new_bottle_name != cxutils.sanitize_bottlename(self.new_bottle_name):
                result['errors'].append(_("Invalid bottle name '%s'") % cxutils.html_escape(self.bottlename))
            elif self.create_new_bottle and self.new_bottle_name != bottlequery.unique_bottle_name(
                    cxutils.sanitize_bottlename(self.new_bottle_name)):
                result['errors'].append(_("There is already a bottle named '%s'") % cxutils.html_escape(self.bottlename))
            else:
                # FIXME: 32-bit bottle hack
                if self.application_name and self.profile and \
                   'Microsoft Office' in self.application_name and \
                   not self.profile.app_profile.preferred_bottle_type and \
                   not cxproduct.get_config_boolean("OfficeSetup", "Enable32BitBottles", False):
                    result['errors'].append("'%s' needs to be installed in a 32-bit bottle. Please enable them through the preferences dialog." % self.application_name)
                elif self.target_bottle.bottlename:
                    if self.application_name:
                        if self.target_bottle.compatible:
                            result['notes'].append(_("CrossOver will install '%(application)s' into the '%(bottlename)s' bottle") % {
                                'application': self.application_name, 'bottlename': cxutils.html_escape(self.target_bottle.bottlename)})
                        else:
                            result['warnings'].append(_("CrossOver will install '%(application)s' into the '%(bottlename)s' bottle. This bottle may be incompatible") % {
                                'application': self.application_name, 'bottlename': cxutils.html_escape(self.target_bottle.bottlename)})
                    else:
                        if self.target_bottle.compatible:
                            result['notes'].append(_("CrossOver will install this software into the '%(bottlename)s' bottle") % {
                                'bottlename': cxutils.html_escape(self.target_bottle.bottlename)})
                        else:
                            result['warnings'].append(_("CrossOver will install this software into the '%(bottlename)s' bottle. This bottle may be incompatible") % {
                                'bottlename': cxutils.html_escape(self.target_bottle.bottlename)})
                else:
                    template_name = bottlemanagement.get_template_name(self.target_bottle.template)
                    if self.application_name:
                        if self.target_bottle.compatible:
                            result['notes'].append(_("CrossOver will install '%(application)s' into a new %(template)s bottle named '%(name)s'") % {
                                'application': self.application_name, 'template': cxutils.html_escape(template_name),
                                'name': cxutils.html_escape(self.newBottleName)})
                        else:
                            result['warnings'].append(_("CrossOver will install '%(application)s' into a new %(template)s bottle named '%(name)s'. This bottle may be incompatible") % {
                                'application': self.application_name, 'template': cxutils.html_escape(template_name),
                                'name': cxutils.html_escape(self.newBottleName)})
                    else:
                        if self.target_bottle.compatible:
                            result['notes'].append(_("CrossOver will install this software into a new %(template)s bottle named '%(name)s'") % {
                                'template': cxutils.html_escape(template_name),
                                'name': cxutils.html_escape(self.newBottleName)})
                        else:
                            result['warnings'].append(_("CrossOver will install this software into a new %(template)s bottle named '%(name)s'. This bottle may be incompatible") % {
                                'template': cxutils.html_escape(template_name),
                                'name': cxutils.html_escape(self.newBottleName)})
        elif self.profile and self.has_installer_source and not self.target_bottle.analyzed():
            result['errors'].append(_("The install cannot continue until the '%s' bottle is scanned for dependencies. Please wait.") % cxutils.html_escape(self.bottlename))

        return result

    @cxobjc.namedSelector(b'getLanguageDetails')
    def get_language_details(self):
        result = {'errors': [], 'warnings': [], 'notes': []}

        if self.installerLocale:
            langid = self.installerLocale
            if self.application_name:
                result['notes'].append(_("CrossOver will install the %(language)s version of '%(application)s'") % {
                    'language': cxutils.html_escape(c4profiles.LANGUAGES.get(langid, langid)), 'application': self.application_name})

        return result

    @cxobjc.namedSelector(b'getMiscDetails')
    def get_misc_details(self):
        result = {'errors': [], 'warnings': [], 'notes': []}

        if self.target_bottle and self.target_bottle.analyzed():
            if self.target_bottle.missing_profiles:
                missing = ", ".join(self.target_bottle.missing_profiles)
                result['errors'].append(_("The install cannot continue because one or more profiles it depends on could not be found: %s") % cxutils.html_escape(missing))
            elif self.target_bottle.missing_media:
                missing = ", ".join(self.get_profile_names(self.target_bottle.missing_media))
                result['errors'].append(_("The install cannot continue because no installer could be found for the following dependencies: %s") % cxutils.html_escape(missing))

        if self.installerSource and not self.installWithSteam:
            src_filename = self.installerSource
            if os.path.isdir(src_filename):
                src_filename = None
                if self.profile:
                    installer_profile = self.get_installer_profile()
                    src_filename = cxaiemedia.locate_installer(installer_profile, self.installerSource)

            if src_filename and not distversion.IS_MACOSX:
                permissions = cxutils.check_mmap_permissions(src_filename)
                if permissions == cxutils.PERMISSIONS_NOREAD:
                    result['errors'].append(_("The installer file '%s' cannot be read.") % cxutils.html_escape(src_filename))
                elif permissions == cxutils.PERMISSIONS_NOEXEC:
                    result['warnings'].append(_("The installer file '%s' is on a noexec filesystem. This may cause the install to fail.") % cxutils.html_escape(src_filename))

        return result


@cxobjc.method(InstallTask, 'getMissingDependencies_')
def get_missing_dependencies(bottle):
    if bottle and bottle.can_run_commands and bottle.installed_packages_ready and \
       bottle.profile and bottle.profile.app_profile:
        profile = bottle.profile.copy()
        profile.app_profile.steamid = None

        install_task = InstallTask()
        install_task.add_bottle(bottle)
        install_task.profile = profile
        install_task.bottlename = bottle.name

        return install_task.target_bottle.missing_dependencies

    return []
